package com.flybits.commons.library.utils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;

/**
 * This class represents all the possible functions that involve parsing time and date objects. In
 * most cases this utility is used by internally by the SDK however, it is open to all developers to
 * use. The functions added to this class are:
 * <ul>
 *     <li>Converting an ISO String representation to a {@link Calendar} object using {@link #toCalendarT(String)} </li>
 * </ul>
 *
 * @since 2.1.0
 */
public class TimeUtilities {
    /**
     * Convert an ISO 8601 String representation to a {@link Calendar} object.
     *
     * @param iso8601string The ISO 8601 String representation in the following format
     *                      yyyy-MM-dd HH:mm:ssZ
     *
     * @return The {@link Calendar} object of a ISO 8601 String representation.
     *
     * @throws ParseException If the String representation is not in the valid format. A valid
     * format for the String should be in the following format: yyyy-MM-dd HH:mm:ssZ.
     */
    public static Calendar toCalendarT(final String iso8601string) throws ParseException {
        Calendar calendar = GregorianCalendar.getInstance();
        String s = iso8601string.replace("Z", "+00:00");
        try {
            s = s.substring(0, 22) + s.substring(23);  // to get rid of the ":"
        } catch (IndexOutOfBoundsException e) {
            throw new ParseException("Invalid length", 0);
        }
        Date date = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ", Locale.CANADA).parse(s);
        calendar.setTime(date);
        return calendar;
    }
}
