package com.flybits.internal;


import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

public class Debouncer {

    private static Debouncer mInstance = null;
    private final ScheduledExecutorService sched = Executors.newScheduledThreadPool(1);
    private final ConcurrentHashMap<Debounce, TimerTask> delayedMap = new ConcurrentHashMap<>();
    private final int interval = 30;
    private boolean reRun;

    private Debouncer(){}

    public static Debouncer getInstance() {
        if (mInstance == null) {
            synchronized (Debouncer.class){
                if (mInstance == null){
                    mInstance = new Debouncer();
                }
            }
        }
        return mInstance;
    }

    public void call(Debounce task){

        if (!delayedMap.containsKey(task)){
            TimerTask timerTask = new TimerTask(task);
            delayedMap.put(task, timerTask);
            task.doSomething();
            sched.schedule(timerTask, interval, TimeUnit.SECONDS);
        }else{
            reRun   = true;
        }
    }

    public void cancel(Debounce task){

        if (delayedMap.containsKey(task)){
            reRun   = false;
        }
    }

    // The task that wakes up when the wait time elapses
    private class TimerTask implements Runnable {
        private final Debounce event;
        private final Object lock = new Object();

        TimerTask(Debounce event) {
            this.event = event;
        }

        public void run() {
            synchronized (lock) {

                if (reRun) {
                    reRun = false;
                    event.doSomething();
                    sched.schedule(this, interval, TimeUnit.SECONDS);
                }else {
                    delayedMap.remove(event);
                }
            }
        }
    }

}
