package com.flybits.commons.library.api

import android.content.Context
import com.flybits.commons.library.models.User

/**
 *  Responsible for [FlybitsScope] Management for [FlybitsManager].
 */
internal object ScopeOfficer {

    @JvmStatic
    var flybitsScopes: HashMap<String, FlybitsScope> = HashMap()

    /**
     * Add a {@link FlybitsScope} to the {@link FlybitsManager}. The {@link FlybitsScope}
     * defines the SDK and all the properties associated to it.
     *
     * @param scope The {@link FlybitsScope} that is associated to this application.
     * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
     * be set.
     **/
    @JvmStatic
    fun addScope(scope: FlybitsScope) {
        flybitsScopes[scope.scopeName] = scope
    }

    /**
     * Remove a {@link FlybitsScope} to the {@link FlybitsManager}. The {@link FlybitsScope}
     * defines the SDK and all the properties associated to it.
     *
     * @param scope The {@link FlybitsScope} that is associated to this application.
     * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
     * be set.
     **/
    @JvmStatic
    fun removeScope(scope: FlybitsScope) {
        flybitsScopes.remove(scope.scopeName)
    }

    /**
     * Clear all scopes {@link FlybitsScope} added to {@link FlybitsManager}. The {@link FlybitsScope}
     * defines the SDK and all the properties associated to it.
     *
     **/
    @JvmStatic
    internal fun clearScopes() {
        flybitsScopes.clear()
    }

    /**
     * Indicates that the login was successfully made and that the appropriate actions for an SDK
     * should be performed.
     *
     * @param context The current Context of the application
     * @param user The logged in {@link User}.
     */
    fun onConnected(context: Context, user: User) {
        for (scopesName in flybitsScopes.keys) {
            flybitsScopes[scopesName]?.onConnected(context, user)
        }
    }

    /**
     * Indicates that the logout was successful. The SDK will need to make the appropriate actions.
     *
     * @param context The current Context of the application.
     * @param jwtToken The JWT token associated to the login user.
     */
    fun onDisconnected(context: Context, jwtToken: String) {
        for (scopesName in flybitsScopes.keys) {
            flybitsScopes[scopesName]?.onDisconnected(context, jwtToken)
        }
    }

    /**
     * Indicates the logged in {@link User} has requested their account to be disabled. Once
     * successful then the SDK will need to perform the appropriate action.
     *
     * @param context The current Context of the application
     * @param jwtToken The JWT token associated to the login user.
     */
    fun onAccountDestroyed(context: Context, jwtToken: String) {
        for (scopesName in flybitsScopes.keys) {
            flybitsScopes[scopesName]?.onAccountDestroyed(context, jwtToken)
        }
    }

    /**
     * Indicates the logged in {@link User} has changed the opted state.
     *
     * @param context The current Context of the application
     * @param optedState True if the user opted in, false if the user opted out.
     */
    fun onOptedStateChange(context: Context, optedState: Boolean) {
        for (scopesName in flybitsScopes.keys) {
            flybitsScopes[scopesName]?.onOptedStateChange(context, optedState)
        }
    }
}