package com.flybits.commons.library.api

import kotlinx.coroutines.*

/**
 * Responsible for notifying subscribed [OptInStateChangeListener]s about
 * changed to the opt in state of the [User].
 *
 * Introduced as a temporary solution to [FlyAway] having no reference to [FlybitsScope]s.
 */
object OptInStateObservable {

    //Stores listener and a boolean value representing last state the listener was notified, null if not notified about any state yet
    private val subscribers: HashMap<OptInStateChangeListener, Boolean?> = hashMapOf()

    /**
     * Subscribe to notifications about the opt in state.
     *
     * @param subscriber [OptInStateChangeListener] subscribing to updates.
     *
     */
    fun subscribe(subscriber: OptInStateChangeListener) {
        subscribers[subscriber] = null
    }

    /**
     * Unsubscribe from notifications about the opt in state.
     *
     * @param subscriber [OptInStateChangeListener] un-subscribing from updates.
     */
    fun unsubscribe(subscriber: OptInStateChangeListener) {
        subscribers.remove(subscriber)
    }

    /**
     * Clears all the subscribers.
     */
    internal fun clear() {
        subscribers.clear()
    }


    /**
     * Broadcast change opt-in state change to all subscribers. Will be launched using
     * [Dispatchers].Default [CoroutineScope].
     *
     * Subscribers will only be alerted about the same change in state once. If two opt outs occur
     * without an opt-in between them, then `onOptedInStateChange()` will be invoked once.
     *
     * @param userOptedIn New opt-in state.
     * @param coroutineScope The CoroutineScope that the broadcast will launch on. By default
     * or if null it will be assigned [Dispatchers.Default]. Parameter nullable for Java interoperability.
     *
     */
    @JvmName("broadcastOptInState")
    internal fun broadcastOptInState(userOptedIn: Boolean, coroutineScope: CoroutineScope? = CoroutineScope(Dispatchers.Default)) {
        (coroutineScope ?: CoroutineScope(Dispatchers.Default)).launch {
            subscribers.forEach {
                //Only notify listener if current broadcast different from last
                if (it.value != userOptedIn) {
                    it.key.onOptedInStateChange(userOptedIn)
                    subscribers[it.key] = userOptedIn
                }
            }
        }
    }
}