package com.flybits.commons.library.api.results

import android.os.Handler
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.http.RequestStatus
import com.flybits.commons.library.models.internal.Result
import java.util.concurrent.ExecutorService
import java.util.concurrent.Executors

/**
 * The [BasicResult] class is used to query the server when no object is expected to be
 * returned, an empty [BasicResultCallback.onSuccess] method or [BasicResultCallback.onException]
 * with an exception as the reason for the failure will be thrown.
 *
 * @param service The [ExecutorService] that is used to run the network request thread.
 * @param handler The handler that allows methods to be run in the UI.
 * @param callback The [BasicResultCallback] that indicates whether or not the network
 *                 request was successful or not.
 */
class BasicResult(val callback: BasicResultCallback? = null,
                  val handler: Handler? = null,
                  service: ExecutorService = Executors.newSingleThreadExecutor())
    : FlybitsResult(service) {

    /**
     * Sets the [FlybitsException] that was thrown when requesting more items to be loaded.
     *
     * @param e The {@link FlybitsException} that caused the network to fail.
     * @return true if the error has been set, false otherwise
     */
    fun setFailed(e: FlybitsException): Boolean {
        return if (failed()) {
            if (callback != null) {
                handler?.post { callback.onException(e) } ?: callback.onException(e)
            }
            true
        } else false
    }

    /**
     * Sets the result of the network request as successful.
     *
     * @return true if the request was successful, false otherwise
     */
    fun setSuccess(): Boolean {
        return if (success()) {
            if (callback != null) {
                handler?.post { callback.onSuccess() } ?: callback.onSuccess()
            }
            true
        } else false
    }

    /**
     * Set the [Result] of the network request.
     *
     * @param result The [Result] of the network request.
     * @return true if the request was successful, false if the request failed
     */
    fun setResult(result: Result<*>) : Boolean {
        return if (result.status == RequestStatus.COMPLETED) {
            setSuccess()
        } else setFailed(result.exception)
    }
}

