package com.flybits.commons.library

import android.content.Context
import com.flybits.commons.library.logging.Logger
import com.flybits.internal.db.CommonsDatabase
import devliving.online.securedpreferencestore.DefaultRecoveryHandler
import devliving.online.securedpreferencestore.SecuredPreferenceStore
import java.lang.Exception

/**
 * Factory object used for creating [SharedElements]. Depending on whether or not encryption
 * is supported on the device, [SecuredSharedElements] or [UnsecuredSharedElements] will
 * be returned by the [SharedElementsFactory.get] method.
 */
object SharedElementsFactory {

    @Volatile
    internal var savedInstance: SharedElements? = null

    /**
     * Get instance of [SharedElements].
     *
     * @param context Context of the application.
     *
     * @return [SecuredSharedElements] if encryption is supported. [UnsecuredSharedElements] otherwise.
     */
    fun get(context: Context): SharedElements {
        return savedInstance ?: synchronized(this) {
            val userDAO = CommonsDatabase.getDatabase(context).userDao()
            savedInstance ?: try {
                Logger.setTag(SharedElements.TAG).d("get(): Creating first instance of SharedElements")
                SecuredPreferenceStore.init(context, SharedElements.FLYBITS_STORAGE_ENC_V2, null
                        , null, DefaultRecoveryHandler())
                SecuredSharedElements(SecuredPreferenceStore.getSharedInstance(), userDAO)
                        .also { savedInstance = it }
            } catch (e: Exception) {
                Logger.exception("SharedElementsFactory.INSTANCE.get()", e)
                UnsecuredSharedElements(context.getSharedPreferences(
                        SharedElements.FLYBITS_STORAGE_UNENCRYPTED_V2, Context.MODE_PRIVATE), userDAO).also { savedInstance = it }
            }
        }
    }

}