package com.flybits.commons.library.api.idps;


import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;

import com.flybits.commons.library.SharedElementsFactory;
import com.flybits.commons.library.api.FlybitsManager;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.HttpDefaultClass;
import com.flybits.commons.library.models.User;
import com.flybits.commons.library.models.internal.Result;

import org.jetbrains.annotations.NotNull;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * The {@code FlybitsIDP} class is responsible for creating a specific IDentity Provider (IDP) that
 * is managed by Flybits. The Flybits IDP creates a {@link com.flybits.commons.library.models.User}
 * based on their First and Last name as well as an email and password. Once the user is registered
 * they will have to confirm through an email that is sent to their entered email address.
 */
public class FlybitsIDP extends IDP{

    private static final String AUTHENTICATE_ENDPOINT       = "/authenticate";
    private static final String RESET_PASSWORD_ENDPOINT     = FlybitsManager.AUTHENTICATION_API + "/sendResetPasswordEmail";
    private static final String SEND_CONFIRMATION           = FlybitsManager.AUTHENTICATION_API + "/sendConfirmEmail";

    private String firstName;
    private String lastName;
    private String email;
    private String password;
    private boolean sendConfirmationEmail;

    private FlybitsIDP(Parcel in) {
        super(in);
        firstName   = in.readString();
        lastName    = in.readString();
        email       = in.readString();
        password    = in.readString();
    }

    /**
     * Constructor used to register a person to the Flybits IDP.
     *
     * @param email The unique email that is used to associated to a specific
     *              {@link com.flybits.commons.library.models.User} within the Flybits ecosystem.
     * @param password The password that is used to log the Flybits
     *                 {@link com.flybits.commons.library.models.User} into the Flybits ecosystem.
     * @param firstName The first name of the {@link com.flybits.commons.library.models.User}.
     * @param lastName The last name of the {@link com.flybits.commons.library.models.User}.
     * @param sendConfirmationEmail indicates whether or not to send a confirmation email.
     */
    public FlybitsIDP(@NonNull String email, @NonNull String password, String firstName, String lastName, boolean sendConfirmationEmail){
        this (email, password, firstName, lastName);
        this.sendConfirmationEmail  = sendConfirmationEmail;
    }

    /**
     * Constructor used to register a person to the Flybits IDP.
     *
     * @param email The unique email that is used to associated to a specific
     *              {@link com.flybits.commons.library.models.User} within the Flybits ecosystem.
     * @param password The password that is used to log the Flybits
     *                 {@link com.flybits.commons.library.models.User} into the Flybits ecosystem.
     * @param firstName The first name of the {@link com.flybits.commons.library.models.User}.
     * @param lastName The last name of the {@link com.flybits.commons.library.models.User}.
     */
    public FlybitsIDP(@NonNull String email, @NonNull String password, String firstName, String lastName){
        this (email, password);
        this.firstName  = firstName;
        this.lastName   = lastName;
    }

    /**
     * Constructor used to login a person to the Flybits IDP.
     *
     * @param email The unique email that is used to associated to a specific
     *              {@link com.flybits.commons.library.models.User} within the Flybits ecosystem.
     * @param password The password that is used to log the Flybits
     *                 {@link com.flybits.commons.library.models.User} into the Flybits ecosystem.
     */
    public FlybitsIDP(@NonNull String email, @NonNull String password){
        this ();
        this.email      = email;
        this.password   = password;
    }

    private FlybitsIDP(){
        super("flybits");
    }

    public String getEmail(){
        return email;
    }

    /**
     * Indicates whether or not a email should be sent automatically after registration
     *
     * @return true if an email should be sent automatically, false otherwise.
     */
    public boolean isSendConfirmationEmail(){
        return sendConfirmationEmail;
    }

    @Override
    public String getAuthenticationEndPoint() {
        return AUTHENTICATE_ENDPOINT;
    }

    @Override
    public JSONObject getBody() {
        JSONObject bodyObject   = new JSONObject();

        try {
            bodyObject.put("email", email);
        }catch (JSONException e){}
        try {
            bodyObject.put("password", password);
        }catch (JSONException e){}

        if (firstName != null && lastName != null) {
            try {
                bodyObject.put("firstName", firstName);
            } catch (JSONException e) {}

            try {
                bodyObject.put("lastName", lastName);
            } catch (JSONException e) {}
        }

        return bodyObject;
    }

    /**
     * Method used to request a new password for a {@link com.flybits.commons.library.models.User}
     * who has signed up to the Flybits IDP.
     *
     * @param context The current state of the application.
     * @param email The email that was used to register a
     *              {@link com.flybits.commons.library.models.User} to the Flybits IDP.
     * @param callback The {@link BasicResultCallback} that indicates whether or not the request was
     *                 successful.
     * @param handler The {@link Handler} that the callback will be invoked on.
     *
     * @return {@link BasicResult} that represents the network request for this query.
     */
    public static BasicResult requestNewPassword(@NonNull final Context context,
                                                 @NonNull final String email,
                                                 @NonNull final BasicResultCallback callback,
                                                 @NonNull final Handler handler) {

        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(callback, handler, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {

                String body     = "{\"email\" : \""+email+ "\"}";
                String url      = SharedElementsFactory.INSTANCE.get(context).getGatewayURL() + RESET_PASSWORD_ENDPOINT;
                try{
                    final Result result = new HttpDefaultClass.Builder(context, false, url).post(body).getResponse();
                    request.setResult(result);
                }catch (final Exception e) {
                    if (e instanceof FlybitsException) {
                        request.setFailed((FlybitsException) e);
                    } else {
                        request.setFailed(new FlybitsException(e));
                    }
                }
            }
        });
        return request;
    }

    /**
     * Method used to request a new password for a {@link com.flybits.commons.library.models.User}
     * who has signed up to the Flybits IDP.
     *
     * @param context The current state of the application.
     * @param email The email that was used to register a
     *              {@link com.flybits.commons.library.models.User} to the Flybits IDP.
     * @param callback The {@link BasicResultCallback} that indicates whether or not the request was
     *                 successful.
     *
     * @return {@link BasicResult} that represents the network request for this query.
     */
    public static BasicResult requestNewPassword(@NonNull final Context context,
                                                 @NonNull final String email,
                                                 @NonNull final BasicResultCallback callback) {
        return requestNewPassword(context, email, callback, new Handler(Looper.getMainLooper()));
    }

    /**
     * Method used to request a new password for a {@link com.flybits.commons.library.models.User}
     * who has signed up to the Flybits IDP.
     *
     * @param context The current state of the application.
     * @param email The email that was used to register a
     *              {@link com.flybits.commons.library.models.User} to the Flybits IDP.
     * @param callback The {@link BasicResultCallback} that indicates whether or not the request was
     *                 successful.
     *
     * @return {@link BasicResult} that represents the network request for this query.
     */
    public static BasicResult sendConfirmationEmail(@NonNull final Context context,
                                                    @NonNull final String email, final BasicResultCallback callback) {
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(callback, handler, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {

                String body     = "{\"email\" : \""+email+ "\"}";
                String url      = SharedElementsFactory.INSTANCE.get(context).getGatewayURL() + SEND_CONFIRMATION;
                try{
                    final Result result = new HttpDefaultClass.Builder(context, false, url)
                            .post(body).getResponse();
                    request.setResult(result);
                }catch (final Exception e) {
                    if (e instanceof FlybitsException) {
                        request.setFailed((FlybitsException) e);
                    } else {
                        request.setFailed(new FlybitsException(e));
                    }
                }
            }
        });
        return request;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    public void writeToParcel(Parcel out, int flags) {
        super.writeToParcel(out, flags);
        out.writeString(firstName);
        out.writeString(lastName);
        out.writeString(email);
        out.writeString(password);
    }

    @Override
    public void onAuthenticated(Context context, @NotNull User user) {
        if (sendConfirmationEmail) {
            FlybitsIDP.sendConfirmationEmail(context, this.getEmail(), null);
        }
    }

    public static final Parcelable.Creator<FlybitsIDP> CREATOR = new Creator<FlybitsIDP>() {
        @Override
        public FlybitsIDP createFromParcel(Parcel in) {
            return new FlybitsIDP(in);
        }

        @Override
        public FlybitsIDP[] newArray(int size) {
            return new FlybitsIDP[size];
        }
    };
}
