package com.flybits.commons.library.api;

import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;

import com.flybits.commons.library.BuildConfig;
import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.SharedElementsFactory;
import com.flybits.commons.library.analytics.Analytics;
import com.flybits.commons.library.api.idps.FlybitsIDP;
import com.flybits.commons.library.api.idps.HttpMethod;
import com.flybits.commons.library.api.idps.IDP;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.ConnectionResult;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ConnectionResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.deserializations.DeserializeLogin;
import com.flybits.commons.library.deserializations.DeserializePagedResponse;
import com.flybits.commons.library.deserializations.DeserializeProject;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.exceptions.InvalidFlybitsManagerException;
import com.flybits.commons.library.exceptions.InvalidRegionException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.Project;
import com.flybits.commons.library.models.User;
import com.flybits.commons.library.models.UserOptedIn;
import com.flybits.commons.library.models.internal.PagedResponse;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.commons.library.models.results.ProjectsResult;
import com.flybits.commons.library.utils.ProjectParameters;
import com.flybits.internal.db.CommonsDatabase;
import com.flybits.internal.db.UserDAO;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * The {@code FlybitsManager} class is responsible for defining all {@link FlybitsScope}s as well as
 * performing all {@link User} operations such as logging in, logging out, disabling user's account,
 * user forgotten password, and refreshing the logged in user's JWT.
 */
public class FlybitsManager {

    public static final String AUTHENTICATION_API      = "/sso/auth";
    static final String PROJECTS_API            = "/kernel/projects";
    static final String DISCONNECT_ENDPOINT     = AUTHENTICATION_API+"/logout";
    static final String ACTIVE_API              = AUTHENTICATION_API+"/activationStatus";

    public static boolean IS_DEBUG = false;

    private Context context;
    private IDP idProvider;
    private Analytics analytics;
    private ArrayList<String> languageCodes;
    private Set<FlybitsScope> listOfScopes;
    private String projectId;
    private SharedElements sharedElements;

    /**
     * Constructor used to define the {@code FlybitsManager} based on the {@link Builder}
     * attributes.
     *
     * @param builder The {@link Builder} that defines all the attributes about the
     * {@code FlybitsManager}.
     */
    private FlybitsManager(Builder builder){
        listOfScopes                = builder.listOfScopes;
        context                     = builder.mContext;
        languageCodes               = builder.languageCodes;
        idProvider                  = builder.idProvider;
        projectId                   = builder.projectId;
        analytics                   = new Analytics(context);
        sharedElements              = SharedElementsFactory.INSTANCE.get(context);

        if (projectId != null){
            Logger.setTag(FlybitsManager.class.getSimpleName()).d("Migrating, and setting project id");
            Map<String, String> args = new HashMap<>();
            args.put(SharedElements.ARG_PROJECT_ID, projectId);
            sharedElements.migrateData(context, args);
            sharedElements.setProjectID(projectId);
        }

        if (languageCodes.size() == 0) {
            languageCodes.add("en");
        }

        //TODO: Get rid of this once FlyAway is able to properly reference scopes and alert them on 403 return of OptOut
        //This will listen for changes to opt in state and make the required updates, executes on background thread
        //Note: If the client creates several instances of FlybitsManager then this will happen many times(singleton?)
        OptInStateObservable.INSTANCE.subscribe(optedIn -> {
            for (FlybitsScope s: listOfScopes) {
                s.onOptedStateChange(context, optedIn);
            }
            UserDAO userDAO = CommonsDatabase.getDatabase(context).userDao();
            User user = userDAO.getSingle();
            if (user != null) {
                user.setOptedIn(optedIn);
                userDAO.update(user);
            }

        });

        String gatewayUrl;
        if (builder.gatewayUrl != null) {
            gatewayUrl = builder.gatewayUrl; //Set to value in builder if present
        } else {
            try{
                android.content.pm.ApplicationInfo ai = context.getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
                Bundle bundle   = ai.metaData;
                if (bundle.containsKey("com.flybits.project.region")){

                    String region   = bundle.getString("com.flybits.project.region");
                    if (region == null) {
                        throw new InvalidRegionException("This version of the SDK only accepts 'south-america' or 'north-america' value as a region in your " +
                                "AndroidManifest.xml file");
                    }else{
                        switch (region) {
                            case "south-america":
                                gatewayUrl = BuildConfig.GATEWAY_SA;
                                break;
                            case "north-america":
                                gatewayUrl = BuildConfig.GATEWAY;
                                break;
                            case "develop":
                                gatewayUrl = BuildConfig.GATEWAY_DEV;
                                break;
                            case "staging":
                                gatewayUrl = BuildConfig.GATEWAY_STAGING;
                                break;
                            default:
                                throw new InvalidRegionException("This version of the SDK only accepts 'south-america' or 'north-america' value as a region in your " +
                                        "AndroidManifest.xml file");
                        }
                    }

                } else {
                    gatewayUrl = BuildConfig.GATEWAY; //No value in AndroidManifest, set to default
                }
            } catch (PackageManager.NameNotFoundException | NullPointerException ex) {
                Logger.exception("FlybitsAPIConstants.getGatewayURL", ex);
                gatewayUrl = BuildConfig.GATEWAY; //Error occurred, set to default
            }
        }

        sharedElements.setGatewayURL(gatewayUrl); //Save in SharedElements
    }

    /**
     * The {@code Builder} class is responsible for setting all the attributes associated to the
     * {@link FlybitsManager}.
     */
    public static final class Builder {

        private final Set<FlybitsScope> listOfScopes;
        private Context mContext;
        private IDP idProvider;
        private ArrayList<String> languageCodes;
        private String projectId;
        private String gatewayUrl;

        /**
         * Constructor that initializes all the objects within the {@code Builder} class.
         *
         * @param context The {@link Context} of the application.
         */
        public Builder(@NonNull Context context) {
            this.mContext       = context;
            this.listOfScopes   = new HashSet<>();
            this.languageCodes  = new ArrayList<>();
            try {
                this.languageCodes.add(Locale.getDefault().getLanguage());
            }catch (Exception e){}
        }

        /**
         * Add a {@link FlybitsScope} to the {@link FlybitsManager}. The {@link FlybitsScope}
         * defines the SDK and all the properties associated to it.
         *
         * @param scope The {@link FlybitsScope} that is associated to this application.
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder addScope(@NonNull FlybitsScope scope) {
            listOfScopes.add(scope);
            return this;
        }

        /**
         * Build the {@link FlybitsManager} object that is used for defining the SDKs that are
         * associated to the application.
         *
         * @return {@link FlybitsManager} object which can be referenced by the SDK to get specific
         * information about the SDKs defined within the application.
         * @throws InvalidFlybitsManagerException if the {@link Builder#addScope(FlybitsScope)}
         * method was not called.
         */
        public FlybitsManager build() throws InvalidFlybitsManagerException {
            checkIfFieldsSet();
            return new FlybitsManager(this);
        }

        /**
         * Sets the {@link IDP} which should be used sign into the Flybits Account Manager.
         *
         * @param idp The {@link IDP} that should be used for registering contextual information to
         *            the account.
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setAccount(IDP idp){
            this.idProvider = idp;
            return this;
        }

        /**
         * Sets the Flybits Android commons SDK in Debug mode. While in debug mode all logs will be
         * displayed within the logcat of your Android Studio including any errors/exceptions.
         *
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setDebug(){
            IS_DEBUG    = true;
            return this;
        }

        /**
         * Sets the Language that the SDK should parse the localized values with.
         * @param languageCode The 2-letter code, such as "en", that indicates which language should
         *                     be used for parsing localized values.
         *
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setLanguage(@NonNull String languageCode){
            languageCodes.clear();
            languageCodes.add(languageCode);
            return this;
        }

        /**
         * Sets the project identifier of the application.
         *
         * @param projectId The unique GUID that represents the Project Id for the application.
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setProjectId(@NonNull String projectId){
            this.projectId = projectId;
            return this;
        }

        /**
         * Optional
         *
         * Set the Gateway/Host URL to be used for communicating with Flybits servers.
         *
         * @param gatewayUrl Gateway/Host URL to be used for communicating with Flybits servers.
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setGatewayURL(@NonNull String gatewayUrl) {
            this.gatewayUrl = gatewayUrl;
            return this;
        }

        private void checkIfFieldsSet()throws InvalidFlybitsManagerException {

            if (listOfScopes.size() == 0){
                throw new InvalidFlybitsManagerException("You must have at least 1 scope set to create a valid FlybitsManager object");
            }

            if (languageCodes.size() != 1 && languageCodes.get(0).length() != 2){
                throw new InvalidFlybitsManagerException("Your language must be a 2-letter code. Make sure you call setLanguage(String)");
            }
        }
    }

    /**
     * Indicates that the logcat should display logs from the SDK.
     */
    public static void setDebug(){
        IS_DEBUG    = true;
    }

    /**
     * Indicates that the logcat should not display logs from the SDK.
     */
    public static void unsetDebug(){
        IS_DEBUG    = false;
    }

    /**
     * Sets the project id to another project, refreshing the current session.
     *
     * @param projectId The project id to bind to.
     * @param callback An interface to resolve successes and failures of this call.
     * @param handler The handler that the callback will be invoked through.
     * @return The {@link BasicResult} is an object that is returned from the Server after a HTTP
     * request is made.
     * @deprecated Please use {@link Project#bindProject(Context, String, BasicResultCallback)}.
     */
    @Deprecated
    public BasicResult bindProject(final String projectId, final BasicResultCallback callback, @NonNull final Handler handler) {
        return Project.bindProject(context, projectId, callback, handler);
    }

    /**
     * Sets the project id to another project, refreshing the current session.
     *
     * @param projectId The project id to bind to.
     * @param callback An interface to resolve successes and failures of this call.
     * @return The {@link BasicResult} is an object that is returned from the Server after a HTTP
     * request is made.
     * @deprecated Please use {@link Project#bindProject(Context, String, BasicResultCallback)}.
     */
    @Deprecated
    public BasicResult bindProject(final String projectId, final BasicResultCallback callback) {
        return Project.bindProject(context, projectId, callback);
    }

    /**
     * Opt the currently connected user out of Flybits. All local data associated to the user
     * will be cleared. All communications with Flybits will be stopped until
     * {@link FlybitsManager#optIn(BasicResultCallback, Handler)} is called.
     *
     * @param callback The callback used to indicate whether or not the opt out was
     * successful or not.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult optOut(final BasicResultCallback callback){
        return optOut(callback, new Handler());
    }

    /**
     * Opt the currently connected user out of Flybits. All local data associated to the user
     * will be cleared. All communications with Flybits will be stopped until
     * {@link FlybitsManager#optIn(BasicResultCallback, Handler)} is called.
     *
     * @param callback The callback used to indicate whether or not the opt out was
     * successful or not.
     * @param handler The handler that the callback result will be posted on.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    BasicResult optOut(final BasicResultCallback callback, final Handler handler){
        return setOptedState(false, callback, handler);
    }

    /**
     * Opt in the currently connected user back into Flybits. All communications with Flybits
     * servers will be restored.
     *
     * @param callback The callback used to indicate whether or not the opt in was
     * successful or not.
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult optIn(final BasicResultCallback callback){
        return optIn(callback, new Handler());
    }

    /**
     * Opt in the currently connected user back into Flybits. All communications with Flybits
     * servers will be restored.
     *
     * @param callback The callback used to indicate whether or not the opt in was
     * successful or not.
     * @param handler The handler that the callback result will be posted on.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    BasicResult optIn(final BasicResultCallback callback, Handler handler){
        return setOptedState(true, callback, handler);
    }

    /**
     * Set the opted state of the connected user out. Opting out(optedState=false) will result in
     * all local data associated to the user being cleared. All communications with Flybits will
     * be stopped until the state is returned to back to opted in. .
     * {@link FlybitsManager#optIn(BasicResultCallback, Handler)} is called.
     *
     * @param optedState True to opt in, false to opt out.
     *
     * @param callback The callback used to indicate whether or not the opt out was
     * successful or not.
     * @param handler Handler that the callback result will be posted on.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    BasicResult setOptedState(final boolean optedState, final BasicResultCallback callback, final Handler handler){
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult query = new BasicResult(callback,handler, executorService);

        executorService.execute(new Runnable() {
            @Override
            public void run() {
                try{
                    Result result = FlyAway.post(context, ACTIVE_API
                            , String.format("{\"%s\":%s}", UserOptedIn.FIELD_IS_ACTIVE, optedState), null
                            , "FlybitsManager.setOptedState", null);
                    if (result.getStatus() == RequestStatus.COMPLETED) {

                        CommonsDatabase commonsDatabase = CommonsDatabase.getDatabase(context);
                        if (!optedState) {
                            //Clear database data & destroy analytics
                            analytics.destroy();
                            commonsDatabase.ctxDataDAO().deleteAll();
                            commonsDatabase.preferenceDAO().clear();
                            commonsDatabase.cachingEntryDAO().clear();
                        } else {
                            analytics.scheduleWorkers();
                        }

                        //Update user model in DB so that the isOptedIn flag is set to optedState passed
                        User user = commonsDatabase.userDao().getSingle();
                        if (user != null) {
                            user.setOptedIn(optedState);
                            commonsDatabase.userDao().update(user);
                        }

                        //TODO: Get rid of this once FlyAway is able to properly reference scopes and alert them on 403 return of OptOut
                        OptInStateObservable.INSTANCE.broadcastOptInState(optedState, null);
                    }
                    query.setResult(result);
                } catch (final FlybitsException e) {
                    query.setResult(new Result(e, "FlybitsManager.optOut() failed."));
                }
            }
        });
        return query;
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method.
     *
     * @param callback The response callback to run after a connection is made or fails.
     * @param handler The handler that the callback will be invoked through.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    BasicResult connect(final BasicResultCallback callback, @NonNull final Handler handler) {
        return connect(callback, true, handler);
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method.
     *
     * @param callback The response callback to run after a connection is made or fails.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult connect(final BasicResultCallback callback) {
        return connect(callback, true);
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method.
     *
     * @param idp The {@link IDP} that should be used for registering contextual information to
     *            the account.
     * @param callback The response callback to run after a connection is made or fails.
     * @param autoUseManifestProject If true, the project id defined in the manifest will be loaded
     *                               automatically.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     * @deprecated Moving forward please use {@link #connect(IDP, BasicResultCallback)}
     */
    @Deprecated
    public BasicResult connect(@NonNull final IDP idp, final BasicResultCallback callback, final boolean autoUseManifestProject) {
        idProvider  = idp;
        return connect(callback, autoUseManifestProject);
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method.
     *
     * @param idp The {@link IDP} that should be used for registering contextual information to the
     *            account.
     * @param callback The response callback to run after a connection is made or fails.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult connect(@NonNull final IDP idp, final BasicResultCallback callback) {
        idProvider  = idp;
        return connect(callback, true);
    }

    BasicResult connect(@NonNull String projectID, @NonNull final IDP idp, final BasicResultCallback callback, @NonNull final Handler handler) {
        SharedElements elements = SharedElementsFactory.INSTANCE.get(context);
        if (!projectID.equals(elements.getProjectID()) && !elements.getSavedJWTToken().equals("")){
            final ExecutorService executorService = Executors.newSingleThreadExecutor();
            final BasicResult query = new BasicResult(callback, handler, executorService);

            FlybitsException e = new FlybitsException("You must disconnect from your previous IDP if " +
                    "you would like to change your Project ID. Your current ProjectID " +
                    "is: " + projectID + " your previously set ProjectID is: " + projectID);
            query.setFailed(e);

            return query;
        }

        idProvider = idp;
        sharedElements.setProjectID(projectID);
        return connect(callback, true, handler);
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method.
     *
     * @param projectID The unique identifier that represents your project.
     * @param idp The {@link IDP} that should be used for registering contextual information to the
     *            account.
     * @param callback The response callback to run after a connection is made or fails.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult connect(@NonNull String projectID, @NonNull final IDP idp, final BasicResultCallback callback) {
        return connect(projectID, idp, callback, new Handler(Looper.getMainLooper()));
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method. If autoUseManifestProject is true,
     * the project to load will be pulled from the manifest. Otherwise it will be in a no-project
     * state. This can only be done with the Flybits IDP.
     *
     * @param callback               The response callback to run after a connection is made or fails.
     * @param autoUseManifestProject If true, the project id defined in the manifest will be loaded
     *                               automatically.
     * @param handler                The handler that the callback will be invoked through.
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    BasicResult connect(final BasicResultCallback callback, final boolean autoUseManifestProject, @NonNull final Handler handler) {

        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult query = new BasicResult(callback, handler, executorService);
        executorService.execute(() -> {
            isConnected(context, true, new ConnectionResultCallback() {
                @Override
                public void onConnected() {
                    FlybitsException e = new FlybitsException("You already have an IDP connected to " +
                            "the Flybits SDK, please call the disconnect method before calling connect again");
                    query.setFailed(e);
                }

                @Override
                public void notConnected() {
                    executorService.execute(() -> {
                        try {
                            sharedElements.setLocalization(languageCodes);

                            if (idProvider != null) {
                                String body = idProvider.getRequestBody(context, autoUseManifestProject).toString();
                                String url = AUTHENTICATION_API + idProvider.getAuthenticationEndPoint();
                                final Result<User> authenticatedUser;
                                HttpMethod requestType = idProvider.getRequestType();
                                if (requestType.equals(HttpMethod.GET)) {
                                    authenticatedUser = FlyAway.get(context, url, idProvider.getHeaders(), new DeserializeLogin(), "FlybitsManager.connect", User.class);
                                } else if (requestType.equals(HttpMethod.PUT)) {
                                    authenticatedUser = FlyAway.put(context, url, body, new DeserializeLogin(), "FlybitsManager.connect", User.class, idProvider.getHeaders());
                                } else {
                                    authenticatedUser = FlyAway.post(context, url, body, new DeserializeLogin(), "FlybitsManager.connect", User.class, idProvider.getHeaders());
                                }


                                if (authenticatedUser.getStatus() == RequestStatus.COMPLETED) {

                                    if (idProvider instanceof FlybitsIDP && ((FlybitsIDP) idProvider).isSendConfirmationEmail()) {
                                        FlybitsIDP.sendConfirmationEmail(context, ((FlybitsIDP) idProvider).getEmail(), null);
                                    }

                                    UserDAO userDAO = CommonsDatabase.getDatabase(context).userDao();
                                    userDAO.delete();
                                    userDAO.insert(authenticatedUser.getResult());
                                    sharedElements.setConnectedIDP(idProvider.getProvider());

                                    analytics.scheduleWorkers();

                                    for (FlybitsScope scope : listOfScopes) {
                                        scope.onConnected(context, authenticatedUser.getResult());
                                    }
                                }
                                query.setResult(authenticatedUser);
                            }
                        } catch (final FlybitsException e) {
                            query.setFailed(e);
                        }
                    });
                }

                @Override
                public void onException(@NonNull FlybitsException exception) {
                    query.setFailed(exception);
                }
            }, handler);
        });
        return query;
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method. If autoUseManifestProject is true,
     * the project to load will be pulled from the manifest. Otherwise it will be in a no-project
     * state. This can only be done with the Flybits IDP.
     *
     * @param callback The response callback to run after a connection is made or fails.
     * @param autoUseManifestProject If true, the project id defined in the manifest will be loaded
     *                               automatically.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult connect(final BasicResultCallback callback, final boolean autoUseManifestProject){

        if (idProvider == null){
            final ExecutorService executorService = Executors.newSingleThreadExecutor();
            final BasicResult query = new BasicResult(callback,new Handler(Looper.getMainLooper()), executorService);

            FlybitsException e = new FlybitsException("You must enter an IDP in the Flybits Builder to use this connect() method.");
            callback.onException(e);

            return query;
        }

        return connect(callback, autoUseManifestProject, new Handler(Looper.getMainLooper()));
    }

    /**
     * Clears all information associated this instance of the SDK. If you use this method
     * persistence between app launches will not be maintained, therefore it is highly recommended
     * that you use the {@link #disconnect(BasicResultCallback)} method over this {@code destroy()}
     * one.
     *
     * @param callback The callback used to indicate whether or not the HTTP request was successful.
     * @param handler The handler that the callback will be invoked through.
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    BasicResult destroy(final BasicResultCallback callback, @NonNull final Handler handler){

        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(callback, handler, executorService);

        executorService.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    analytics.flush(null,false);
                    String jwtToken = sharedElements.getSavedJWTToken();
                    final Result deleteUser = FlyAway.delete(context, User.ME_ENDPOINT, "FlybitsManager.destroy", null);
                    if (deleteUser.getStatus() == RequestStatus.COMPLETED) {

                        clearSDKData(context);
                        for (FlybitsScope scope : listOfScopes) {
                            scope.onAccountDestroyed(context, jwtToken);
                        }
                    }
                    request.setResult(deleteUser);
                } catch (final FlybitsException e) {
                    request.setFailed(e);
                }
            }
        });
        return request;
    }

    /**
     * Clears all information associated this instance of the SDK. If you use this method
     * persistence between app launches will not be maintained, therefore it is highly recommended
     * that you use the {@link #disconnect(BasicResultCallback)} method over this {@code destroy()}
     * one.
     *
     * @param callback The callback used to indicate whether or not the HTTP request was successful.
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult destroy(final BasicResultCallback callback){
        return destroy(callback, new Handler(Looper.getMainLooper()));
    }

    /**
     * The {@code disconnect} method is responsible for clearing the server sessions with Flybits.
     * In addition, once a logout is successful it will notify all the registered
     * {@link FlybitsScope}s through the {@link FlybitsScope#onDisconnected(Context, String)}
     * method.
     *
     * @param callback The callback used to indicate whether or not the disconnection was
     * successful or not.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult disconnect(final BasicResultCallback callback){
        return disconnect(callback, false);
    }

    /**
     * The {@code disconnect} method is responsible for clearing the server sessions with Flybits.
     * In addition, once a logout is successful it will notify all the registered
     * {@link FlybitsScope}s through the {@link FlybitsScope#onDisconnected(Context, String)} method.
     *
     * @param callback The callback used to indicate whether or not the disconnection was
     * successful or not.
     * @param disconnectOnException true if the disconnect should clear all data even if the network
     *                              request was unsuccessful, false otherwise.
     * @param handler The handler that the callback will be invoked through.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    BasicResult disconnect(final BasicResultCallback callback, final boolean disconnectOnException, @NonNull final Handler handler){
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(callback, handler, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {

                String jwtToken = sharedElements.getSavedJWTToken();
                try {
                    analytics.flush(null, false);

                    final Result<String> disconnected = FlyAway.post(context, DISCONNECT_ENDPOINT, "", null, "FlybitsManager.disconnect", null);
                    if (disconnected.getStatus() == RequestStatus.COMPLETED || disconnectOnException) {

                        clearSDKData(context);
                        for (FlybitsScope scope : listOfScopes) {
                            scope.onDisconnected(context, jwtToken);
                        }
                        request.setResult(new Result(200, ""));
                    }else {
                        request.setResult(disconnected);
                    }

                } catch (final FlybitsException e) {
                    if (disconnectOnException) {

                        clearSDKData(context);
                        for (FlybitsScope scope : listOfScopes) {
                            scope.onDisconnected(context, jwtToken);
                        }
                    }
                    request.setFailed(e);
                }
            }
        });
        return request;
    }

    /**
     * The {@code disconnect} method is responsible for clearing the server sessions with Flybits.
     * In addition, once a logout is successful it will notify all the registered
     * {@link FlybitsScope}s through the {@link FlybitsScope#onDisconnected(Context, String)} method.
     *
     * @param callback The callback used to indicate whether or not the disconnection was
     * successful or not.
     * @param handler The handler that the callback will be invoked through.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    BasicResult disconnect(final BasicResultCallback callback, @NonNull final Handler handler){
        return disconnect(callback, false, handler);
    }

    /**
     * The {@code disconnect} method is responsible for clearing the server sessions with Flybits.
     * In addition, once a logout is successful it will notify all the registered
     * {@link FlybitsScope}s through the {@link FlybitsScope#onDisconnected(Context, String)} method.
     *
     * @param callback The callback used to indicate whether or not the disconnection was
     * successful or not.
     * @param disconnectOnException true if the disconnect should clear all data even if the network
     *                              request was unsuccessful, false otherwise.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult disconnect(final BasicResultCallback callback, final boolean disconnectOnException){
        return disconnect(callback, disconnectOnException, new Handler(Looper.getMainLooper()));
    }

    /**
     * Called to get all {@code Project}s that exist on this account.
     * @param params Defines paging info.
     * @param callback The callback used to collect the Project objects, or an error if one occured.
     * @param handler The handler that the callback will be invoked through.
     * @return The {@link ProjectsResult} object can be used to continue paging if more items can
     * be fetched.
     */
    ProjectsResult getProjects(final ProjectParameters params, final PagedResultCallback<Project> callback, @NonNull final Handler handler)
    {
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ProjectsResult result = new ProjectsResult(context, params, callback, executorService, handler, this);

        executorService.execute(new Runnable() {
            @Override
            public void run() {

                try{
                    boolean isConnnected = !sharedElements.getSavedJWTToken().equals("");
                    if (isConnnected && getIDP() instanceof FlybitsIDP){
                        DeserializeProject singleDeserializaer = new DeserializeProject();
                        DeserializePagedResponse<Project> deserializer = new DeserializePagedResponse<Project>(singleDeserializaer);
                        final Result<PagedResponse<Project>> projectList = FlyAway.get(context, PROJECTS_API, params, deserializer, "FlybitsManager.getProjects");
                        result.setResult(projectList);
                    }
                    else {
                        throw new FlybitsException("Either not yet connected to Flybits, or you are not using the Flybits IDP. GetProjects() only works with the Flybits IDP.");
                    }
                }catch (final FlybitsException e){
                    result.setResult(new Result(e, ""));
                }
            }
        });

        return result;
    }

    /**
     * Called to get all {@code Project}s that exist on this account.
     * @param params Defines paging info.
     * @param callback The callback used to collect the Project objects, or an error if one occured.
     * @return The {@link ProjectsResult} object can be used to continue paging if more items can
     * be fetched.
     */
    public ProjectsResult getProjects(final ProjectParameters params, final PagedResultCallback<Project> callback)
    {
        return getProjects(params, callback, new Handler(Looper.getMainLooper()));
    }

    /**
     * The {@code getUser} method is responsible for retrieving user information about the Flybits
     * {@link User}. The {@link User} contains basic personal attributes such as
     * {@link User#getFirstName()} and {@link User#getLastName()} as well as Flybits information such as
     * {@link User#getId()} and {@link User#getDeviceID()}.
     * @param context The context of the activity that is calling this method.
     * @return The {@link ObjectResult} is an object that is returned from the Server after a HTTP
     * request is made.
     * @deprecated Please use the more optimized {@link User#getSelf} static method.
     */
    @Deprecated
    public static ObjectResult<User> getUser(final Context context){
        return User.getSelf(context, null);
    }

    /**
     * The {@code getUser} method is responsible for retrieving user information about the Flybits
     * {@link User}. The {@link User} contains basic personal attributes such as
     * {@link User#getFirstName()} and {@link User#getLastName()} as well as Flybits information such as
     * {@link User#getId()} and {@link User#getDeviceID()}.
     * @param context The context of the activity that is calling this method.
     * @param callback Indicates whether or not the network request was successfully made.
     * @param handler The handler that the callback will be invoked through.
     * @return The {@link ObjectResult} is an object that is returned from the Server after a HTTP
     * request is made.
     * @deprecated Please use the more optimized {@link User#getSelf} static method.
     */
    @Deprecated
    static ObjectResult<User> getUser(final Context context, ObjectResultCallback<User> callback, @NonNull final Handler handler){
        return User.getSelf(context, callback, handler);
    }

    /**
     * The {@code getUser} method is responsible for retrieving user information about the Flybits
     * {@link User}. The {@link User} contains basic personal attributes such as
     * {@link User#getFirstName()} and {@link User#getLastName()} as well as Flybits information such as
     * {@link User#getId()} and {@link User#getDeviceID()}.
     * @param context The context of the activity that is calling this method.
     * @param callback Indicates whether or not the network request was successfully made.
     * @return The {@link ObjectResult} is an object that is returned from the Server after a HTTP
     * request is made.
     * @deprecated Please use the more optimized {@link User#getSelf} static method.
     */
    @Deprecated
    public static ObjectResult<User> getUser(final Context context, ObjectResultCallback<User> callback){
        return User.getSelf(context, callback);
    }

    /**
     * Check to see if the SDK currently has established a connection with the Flybits Server.
     *
     * @param context The Context of the application.
     * @param confirmThroughNetwork Indicates whether or not a confirm should take place with the
     *                              server. If false, this will simply check is a token is saved
     *                              locally, this token may however not be valid. If true, it will
     *                              confirm with the server that it is valid.
     * @param callback The callback that indicates whether or not the SDK is currently connected to
     *                 Flybits.
     * @param handler The handler that the callback will be invoked through.
     *
     * @return The {@link ConnectionResult} which is executing the network request (if any) to
     * determine whether or not the connection is valid.
     */
    static ConnectionResult isConnected(@NonNull final Context context,
                                               final boolean confirmThroughNetwork,
                                               @NonNull final ConnectionResultCallback callback,
                                               @NonNull final Handler handler) {

        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        ConnectionResult query   = new ConnectionResult(callback, handler, executorService);
        final SharedElements sharedElements = SharedElementsFactory.INSTANCE.get(context);
        executorService.execute(new Runnable() {
            @Override
            public void run() {

                if (confirmThroughNetwork && !sharedElements.getSavedJWTToken().equals("")) {

                    try {
                        final FlyJWT jwt = FlyJWT.Companion.get(context);
                        final Result getJWT = jwt.refresh();
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                if (getJWT.getStatus() == RequestStatus.NOT_CONNECTED) {
                                    callback.notConnected();
                                } else if (getJWT.getStatus() == RequestStatus.COMPLETED) {
                                    callback.onConnected();
                                } else {
                                    callback.onException(getJWT.getException());
                                }
                            }
                        });
                    } catch (final FlybitsException e) {
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                callback.onException(e);
                            }
                        });
                    }
                } else if (sharedElements.getSavedJWTToken().equals("")) {
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.notConnected();
                        }
                    });

                } else {
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.onConnected();
                        }
                    });
                }
            }
        });
        return query;
    }

    /**
     * Check to see if the SDK currently has established a connection with the Flybits Server.
     *
     * @param context The Context of the application.
     * @param confirmThroughNetwork Indicates whether or not a confirm should take place with the
     *                              server. If false, this will simply check is a token is saved
     *                              locally, this token may however not be valid. If true, it will
     *                              confirm with the server that it is valid.
     * @param callback The callback that indicates whether or not the SDK is currently connected to
     *                 Flybits.
     * @return The {@link ConnectionResult} which is executing the network request (if any) to
     * determine whether or not the connection is valid.
     */
    public static ConnectionResult isConnected(@NonNull final Context context,
                                               final boolean confirmThroughNetwork,
                                               @NonNull final ConnectionResultCallback callback){

        return isConnected(context, confirmThroughNetwork, callback, new Handler(Looper.getMainLooper()));
    }

    IDP getIDP(){
        return idProvider;
    }

    Set<FlybitsScope> getScopes(){
        return listOfScopes;
    }

    /**
     * Clear all SDK data that is tied to the currently authenticated user.
     *
     * @param context Context associated with the application
     */
    void clearSDKData(Context context) {
        analytics.destroy();
        SharedElements sharedElements = SharedElementsFactory.INSTANCE.get(context);
        sharedElements.setJWTToken("");
        sharedElements.setConnectedIDP("");
        CommonsDatabase commonsDatabase = CommonsDatabase.getDatabase(context);
        commonsDatabase.userDao().delete();
        commonsDatabase.ctxDataDAO().deleteAll();
        commonsDatabase.preferenceDAO().clear();
        commonsDatabase.cachingEntryDAO().clear();
    }
}
