package com.flybits.internal.models.preferences;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;

import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.internal.db.CommonsDatabase;
import com.flybits.internal.db.models.Preference;

import org.json.JSONArray;

import java.util.ArrayList;
import java.util.List;

/**
 * The {@code FlybitsFavourite} class is responsible for providing convenience methods that allow
 * the SDKs to easily add, remove, and get favourites. In addition to saving and retrieving
 * {@code Favourites} online this class also allows you to save {@code Favourites} within a local
 * DB which can be accessed at any point.
 */
public class FlybitsFavourite extends FlybitsPreferences{

    /**
     * Default Constructor used to define the context of the {@code Activity} that is calling the
     * {@code Favourites}.
     *
     * @param context The context of the activity.
     */
    public FlybitsFavourite(Context context) {
        super(context, new Handler(Looper.getMainLooper()));
    }

    /**
     * Constructor used to define the context of the {@code Activity} that is calling the
     * {@code Favourites}.
     *
     * @param context The context of the activity.
     * @param handler The {@code Handler} that is used to provide the ability to run code in the
     *                main UI Thread.
     */
    public FlybitsFavourite(Context context, Handler handler) {
        super(context, handler);
    }

    /**
     * Add the String representation of the value to Favourites preferences based on the provided
     * {@code key}.
     *
     * @param key The key that the {@code FlybitsFavourite} should be saved under.
     * @param value The String representation of the value that should be saved. If this value is an
     *              object it would be good to serialize (ex. JSON) the object before calling this
     *              method.
     * @param callback The callback that indicates whether or not the request was successful.
     */
    public void add(final String key, final String value, final BasicResultCallback callback) {
        getExecutorService().execute(new Runnable() {
            @Override
            public void run() {

                if (containsDB(key, value)) {

                    if (getHandler() != null) {
                        getHandler().post(new Runnable() {
                            @Override
                            public void run() {
                                callback.onSuccess();
                            }
                        });
                    }
                    return;
                }

                final Preference preference = new Preference(key, value);
                CommonsDatabase.getDatabase(getContext()).preferenceDAO().insert(preference);
                List<String> listOfItemsSaved = CommonsDatabase.getDatabase(getContext()).preferenceDAO().getIdsByKey(key);

                JSONArray jsonArray = new JSONArray();
                for (String value : listOfItemsSaved) {
                    jsonArray.put(value);
                }

                final Result result = FlybitsFavourite.this.update(key, jsonArray);
                if (getHandler() != null) {
                    if (result.getStatus() == RequestStatus.COMPLETED) {
                        getHandler().post(new Runnable() {
                            @Override
                            public void run() {
                                callback.onSuccess();
                            }
                        });
                    } else {

                        CommonsDatabase.getDatabase(getContext()).preferenceDAO().deleteByKeyAndValue(preference.getPrefKey(), preference.getValue());
                        getHandler().post(new Runnable() {
                            @Override
                            public void run() {
                                callback.onException(result.getException());
                            }
                        });
                    }
                }
            }
        });
    }

    /**
     * Remove the String representation of the value from the Favourites preferences based on the
     * provided {@code key}.
     *
     * @param key The key that the {@code FlybitsFavourite} should be removed from.
     * @param value The String representation of the value that should be removed. If this value is
     *              an object it would be good to serialize (ex. JSON) the object before calling
     *              this method.
     * @param callback The callback that indicates whether or not the request was successful.
     */
    public void remove(final String key, final String value, final BasicResultCallback callback) {
        getExecutorService().execute(new Runnable() {
            @Override
            public void run() {

                if (!containsDB(key, value)) {
                    if (getHandler() != null) {
                        getHandler().post(new Runnable() {
                            @Override
                            public void run() {
                                callback.onSuccess();
                            }
                        });
                    }
                    return;
                }

                CommonsDatabase.getDatabase(getContext()).preferenceDAO().deleteByKeyAndValue(key, value);
                List<String> listOfItemsSaved = CommonsDatabase.getDatabase(getContext()).preferenceDAO().getIdsByKey(key);

                JSONArray jsonArray = new JSONArray();
                for (String value : listOfItemsSaved) {
                    jsonArray.put(value);
                }

                final Result result = FlybitsFavourite.this.update(key, jsonArray);
                if (getHandler() != null) {
                    if (result.getStatus() == RequestStatus.COMPLETED) {
                        getHandler().post(new Runnable() {
                            @Override
                            public void run() {
                                callback.onSuccess();
                            }
                        });
                    } else {
                        Preference preference = new Preference(key, value);
                        CommonsDatabase.getDatabase(getContext()).preferenceDAO().insert(preference);
                        getHandler().post(new Runnable() {
                            @Override
                            public void run() {
                                callback.onException(result.getException());
                            }
                        });
                    }
                }
            }
        });
    }

    /**
     * Clear all the {@code FlybitsFavourite} associated to the {@code key}.
     *
     * @param key The key that the {@code FlybitsFavourite} should be removed from.
     * @param callback The callback that indicates whether or not the request was successful.
     */
    public void clear(final String key, final BasicResultCallback callback) {
        getExecutorService().execute(new Runnable() {
            @Override
            public void run() {

                FlybitsFavourite.this.putStringList(key, new ArrayList<String>(), new BasicResultCallback() {
                    @Override
                    public void onSuccess() {
                        CommonsDatabase.getDatabase(getContext()).preferenceDAO().deleteByPrefKey(key);
                        if (callback != null && getHandler() != null){
                            getHandler().post(new Runnable() {
                                @Override
                                public void run() {
                                    callback.onSuccess();
                                }
                            });
                        }
                    }

                    @Override
                    public void onException(final FlybitsException exception) {
                        if (callback != null && getHandler() != null){
                            getHandler().post(new Runnable() {
                                @Override
                                public void run() {
                                    callback.onException(exception);
                                }
                            });
                        }
                    }
                });
            }
        });
    }

    private boolean containsDB(final String key, final String value){
        return CommonsDatabase.getDatabase(getContext()).preferenceDAO().getIdsByKeyAndValue(key, value).size() > 0;
    }

    /**
     * Check to see if the a specific {@code value} has been already saved as a
     * {@code FlybitsFavourite}.
     *
     * @param key The key that the {@code FlybitsFavourite} should be checked.
     * @param value The String representation of the value that should be check to see if it is
     *              saved. If this value is an object it would be good to serialize (ex. JSON) the
     *              object before calling this method.
     * @param callback The callback that indicates whether or not the request was successful.
     */
    public void containsDB(final String key, final String value, @NonNull final ObjectResultCallback<Boolean> callback){
        getExecutorService().execute(new Runnable() {
            @Override
            public void run() {
                if (callback != null) {
                    final List<Preference> items = CommonsDatabase.getDatabase(getContext()).preferenceDAO().getIdsByKeyAndValue(key, value);
                    if (getHandler() != null){
                        getHandler().post(new Runnable() {
                            @Override
                            public void run() {
                                callback.onSuccess(items.size() > 0);
                            }
                        });
                    }
                }
            }
        });
    }

    /**
     * Get the {@code FlybitsFavourite} values that are associated to the {@code key} from the local
     * DB.
     *
     * @param key The key that the {@code FlybitsFavourite} should be retrieved.
     * @param callback The callback that indicates whether or not the request was successful.
     */
    public void getFromDB(final String key, @NonNull final ObjectResultCallback<List<String>> callback){
        getExecutorService().execute(new Runnable() {
            @Override
            public void run() {
                if (callback != null) {
                    final List<String> items = CommonsDatabase.getDatabase(getContext()).preferenceDAO().getIdsByKey(key);
                    if (getHandler() != null){
                        getHandler().post(new Runnable() {
                            @Override
                            public void run() {
                                callback.onSuccess(items);
                            }
                        });
                    }
                }
            }
        });
    }

    /**
     * Get the {@code FlybitsFavourite} values that are associated to the {@code key} from the
     * server.
     *
     * @param key The key that the {@code FlybitsFavourite} should be retrieved.
     * @param callback The callback that indicates whether or not the request was successful.
     */
    public void getFromServer(final String key, @NonNull final ObjectResultCallback<ArrayList<String>> callback){
        FlybitsFavourite.this.getStringList(key, callback);
    }

    /**
     * Sync the local {@code FlybitsFavourite} with those on the Server.
     *
     * @param key The key that the {@code FlybitsFavourite} should be retrieved for.
     * @param callback The callback that indicates whether or not the request was successful.
     */
    public void reset(final String key, final ObjectResultCallback<ArrayList<String>> callback){
        FlybitsFavourite.this.getStringList(key, new ObjectResultCallback<ArrayList<String>>() {
            @Override
            public void onSuccess(final ArrayList<String> items) {
                getExecutorService().execute(new Runnable() {
                    @Override
                    public void run() {
                        CommonsDatabase.getDatabase(getContext()).preferenceDAO().deleteByPrefKey(key);
                        ArrayList<Preference> listOfPreferences = new ArrayList<>();
                        for (String item : items){
                            listOfPreferences.add(new Preference(key, item));
                        }
                        CommonsDatabase.getDatabase(getContext()).preferenceDAO().insert(listOfPreferences);

                        if (callback != null && getHandler() != null){
                            getHandler().post(new Runnable() {
                                @Override
                                public void run() {
                                    callback.onSuccess(items);
                                }
                            });
                        }
                    }
                });
            }

            @Override
            public void onException(final FlybitsException exception) {
                if (callback != null && getHandler() != null){
                    getHandler().post(new Runnable() {
                        @Override
                        public void run() {
                            callback.onException(exception);
                        }
                    });
                }
            }
        });
    }
}
