package com.flybits.commons.library.analytics;

import android.support.annotation.NonNull;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

public class Properties {

    private Map<String, Object> propertyMap = new LinkedHashMap<>();

    /**
     * Standard constructor creating an empty {@link Properties} object.
     */
    public Properties(){}

    /**
     * Creates a {@link Properties} object with the propeties initialized to the
     * map of key/value pairs.
     *
     * @param map A {@link Map} of key/value pairs (string/object) to initialize to.
     */
    public Properties(@NonNull Map<String, Object> map)
    {
        propertyMap = map;
    }

    /**
     * Creates a {@link Properties} object with the propeties initialized to the
     * map of key/value pairs given in json form.
     *
     * @param json A json {@link String} containing key/value pairs (string/object) to initialize to.
     * @throws JSONException If the {@code json} parameter cannot be parsed properly.
     */
    public Properties(@NonNull String json) throws JSONException {
        JSONObject rootObj = new JSONObject(json);
        Iterator<String> iter = rootObj.keys();
        while (iter.hasNext())
        {
            String key = iter.next();
            propertyMap.put(key, rootObj.get(key));
        }
    }

    /**
     * Adds a new numerical property to this {@link Properties} object.
     *
     * @param name The name of the property being added
     * @param value The value of the property.
     *
     * @return This {@link Properties} object, allowing chaining.
     */
    public Properties addProperty(@NonNull String name, int value) {
        propertyMap.put(name, value);
        return this;
    }

    /**
     * Adds a new numerical property to this {@link Properties} object.
     *
     * @param name The name of the property being added
     * @param value The value of the property.
     *
     * @return This {@link Properties} object, allowing chaining.
     */
    public Properties addProperty(@NonNull String name, long value) {
        propertyMap.put(name, value);
        return this;
    }

    /**
     * Adds a new numerical property to this {@link Properties} object.
     *
     * @param name The name of the property being added
     * @param value The value of the property.
     *
     * @return This {@link Properties} object, allowing chaining.
     */
    public Properties addProperty(String name, float value) {
        propertyMap.put(name, value);
        return this;
    }

    /**
     * Adds a new numerical property to this {@link Properties} object.
     *
     * @param name The name of the property being added
     * @param value The value of the property.
     *
     * @return This {@link Properties} object, allowing chaining.
     */
    public Properties addProperty(String name, double value) {
        propertyMap.put(name, value);
        return this;
    }

    /**
     * Adds a new string property to this {@link Properties} object.
     *
     * @param name The name of the property being added
     * @param value The value of the property.
     *
     * @return This {@link Properties} object, allowing chaining.
     */
    public Properties addProperty(@NonNull String name, String value) {
        propertyMap.put(name, value);
        return this;
    }

    /**
     * Returns a map of all properties that were added to the {@link Properties} object.
     *
     * @return A map of all properties added to this {@link Properties} object.
     */
    public Map<String, Object> getMap()
    {
        return propertyMap;
    }

    /**
     * Converts the map of all properties in this {@link Properties} object to JSON form.
     *
     * @return The JSON in string form.
     * @throws JSONException If the {@code json} parameter cannot be parsed properly.
     */
    public String toJSON() throws JSONException {

        JSONObject rootObj = new JSONObject();
        for (Map.Entry<String, Object> entry : propertyMap.entrySet()) {
            rootObj.put(entry.getKey(), entry.getValue());
        }
        return rootObj.toString();

    }
}
