package com.flybits.commons.library.models.internal;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

/**
 * The {@link QueryParameters} class is used to define all common parameters that can be added to
 * any GET request in order to filter the results requested. For example, you may want to add paging
 * to your GET request in order to get a different set of results.
 */
public class QueryParameters {

    private long limit;
    private long offset;

    private String labelsQuery;

    private OrderBy orderBy = null;
    private SortByEnumeratable sortBy = null;

    private String cachingKey;
    private int cachingLimit;

    /**
     * Default constructor that initializes all the local variables including {@code limit} and
     * {@code offset} used for paging, and an empty {@code labels} list for filtering by label.
     *
     * @param builder The {@link QueryBuilder} that is used to construct the GET network query.
     */
    public QueryParameters(QueryBuilder builder) {
        limit           = builder.limit;
        offset          = builder.offset;
        sortBy          = builder.sortBy;
        orderBy         = builder.orderBy;
        cachingKey      = builder.cachingKey;
        cachingLimit    = builder.cachingLimit;

        if (builder.labelsQuery != null){
            this.labelsQuery = builder.labelsQuery;
        }
    }

    /**
     * Get the list of query parameters that should be added to the GET request.
     *
     * @return A list of parameters that should be added to the GET query.
     */
    public Map<String, ArrayList<String>> getQueryParams(){

        Map<String, ArrayList<String>> params = new HashMap<>();

        if (limit > 0 && offset > -1) {

            ArrayList<String> limitParam = new ArrayList<>();
            limitParam.add( String.valueOf(limit));
            params.put("limit", limitParam);

            ArrayList<String> offsetParam = new ArrayList<>();
            offsetParam.add(String.valueOf(offset));
            params.put("offset", offsetParam);
        }

        if (labelsQuery != null) {
            ArrayList<String> labelQueryParam = new ArrayList<>();
            labelQueryParam.add(String.valueOf(labelsQuery));
            params.put("labelsFormula", labelQueryParam);
        }

        if (sortBy != null && orderBy != null) {
            ArrayList<String> sortByParam = new ArrayList<>();
            ArrayList<String> orderByParam = new ArrayList<>();
            sortByParam.add(sortBy.getValue());
            orderByParam.add(orderBy.toString());

            params.put("sortby", sortByParam);
            params.put("sortorder", orderByParam);
        }

        return params;
    }

    /**
     * Gets key that represents the caching of the returned objects. This is useful in the event
     * that multiple of the same APIs are made with different query parameters.
     *
     * @return The unique caching key for the data being stored.
     */
    public String getCachingKey() {
        return cachingKey;
    }

    /**
     * Get maximum number of caching entries that for an entity.
     *
     * @return The number of entries that should be cached.
     */
    public int getCachingLimit() {
        return cachingLimit;
    }

    /**
     * Gets the current pagination limit in these params.
     * @return The current limit.
     */
    public long getLimit() {
        return limit;
    }

    /**
     * Gets the current labels to filter by in these params.
     * @return The labels currently being filtered by.
     */
    public String getLabels() {
        return labelsQuery;
    }

    /**
     * Gets the current pagination offset in these params.
     * @return The current offset.
     */
    public long getOffset() {
        return offset;
    }

    /**
     * Gets the current sort order in these params.
     * @return The current sort order.
     */
    public OrderBy getOrderBy() {
        return orderBy;
    }

    /**
     * Gets the current field being sorted by in these params.
     * @return The current sort field.
     */
    public SortByEnumeratable getSortBy() {
        return sortBy;
    }

     /**
      * Sets paging options, based on a starting offset to some limit.
      *
      * @param limit The maximum number of returned objects.
      * @param offset The offset of where the next X number of response objects will be returned
      *               from where X is the limit.
      */
    public void setPaging(long limit, long offset) {
        /*
        Do not remove this method. It seems like it should be in the QueryBuilder however we need it
        for changing the Paging object when retrieving more entities in the PagedResult class. If
        these were in the same package then this method would be protected
         */
        this.limit  = limit;
        this.offset = offset;
    }
}
