package com.flybits.commons.library.api.results;

import android.content.Context;
import android.os.Handler;
import android.support.annotation.NonNull;

import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.models.internal.Result;

import java.util.concurrent.ExecutorService;

/**
 * The {@code BasicResult} class is used to query the server when no object is expected to be
 * returned, but rather a simple onSuccess or onException with an exception as the reason for the
 * failure.
 */
public class BasicResult extends FlybitsResult {

    private BasicResultCallback callback;
    private Handler handler;
    private boolean isBackwardsCompat;

    /**
     * Constructor used to initialize the {@code BasicResult} object.
     *
     * @param context The context of the application.
     * @param callback The {@link BasicResultCallback} that indicates whether or not the network
     *                 request was successful or not.
     */
    public BasicResult(@NonNull Context context, BasicResultCallback callback){
        super(context);
        this.callback   = callback;
    }

    /**
     * Constructor used to initialize the {@code BasicResult} object.
     *
     * @param context The context of the application.
     * @param callback The {@link BasicResultCallback} that indicates whether or not the network
     *                 request was successful or not.
     * @param service The {@link ExecutorService} that is used to run the network request thread.
     * @deprecated The more optimized {@link #BasicResult(Context, BasicResultCallback, ExecutorService, Handler)}
     * should be used.
     */
    @Deprecated
    public BasicResult(@NonNull Context context, BasicResultCallback callback, @NonNull ExecutorService service){
        super(context, service);
        this.callback   = callback;
    }

    /**
     * Constructor used to initialize the {@code BasicResult} object.
     *
     * @param context The context of the application.
     * @param callback The {@link BasicResultCallback} that indicates whether or not the network
     *                 request was successful or not.
     * @param service The {@link ExecutorService} that is used to run the network request thread.
     * @param handler The handler that allows methods to be run in the UI.
     */
    public BasicResult(@NonNull Context context, BasicResultCallback callback, @NonNull ExecutorService service, @NonNull Handler handler){
        super(context, service);
        this.callback       = callback;
        this.handler        = handler;
        isBackwardsCompat   = true;
    }

    /**
     * Sets the {@link FlybitsException} that was thrown when request more items to be loaded.
     *
     * @param e The {@link FlybitsException} that caused the network to fail.
     * @return true if the error has been set, false otherwise
     */
    @Override
    public boolean setFailed(final FlybitsException e) {
        if (super.setFailed(e)) {
            if (callback != null){
                if (isBackwardsCompat && handler != null){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.onException(e);
                        }
                    });
                }else if (!isBackwardsCompat){
                    callback.onException(e);
                }
            }
            return true;
        }
        return false;
    }

    /**
     * Sets the result of the network request as successful.
     *
     * @return true if the request was successful, false otherwise
     */
    public boolean setSuccess(){
        if (super.setSuccess()){
            if (callback != null){
                if (isBackwardsCompat && handler != null){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.onSuccess();
                        }
                    });
                }else if (!isBackwardsCompat){
                    callback.onSuccess();
                }
            }
            return true;
        }
        return false;
    }

    /**
     * Set the {@link Result} of the network request.
     *
     * @param result The {@link Result} of the network request.
     */
    public void setResult(@NonNull Result result){
        if (result.getStatus() == RequestStatus.COMPLETED){
            setSuccess();
        }else{
            setFailed(result.getException());
        }
    }
}
