package com.flybits.commons.library.utils;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.util.ArrayList;

/**
 * Constructs a Conjunctive Normal Form formula based of any set of Strings. Examples of CNF include
 * the following;
 *
 * (A)
 * (A OR B)
 * (A) AND (B)
 * (A OR B) AND (C)
 * (A OR B OR C) AND (D OR E OR F)
 *
 * Notice that all ORs are grouped together while groups (indicated within a (...) ) are separated
 * by an AND. Something like
 *
 * (A AND B) OR (D AND C)
 *
 * Is NOT CNF! The above should be converted to the following
 *
 * (A OR D) AND (A OR C) AND (B OR C) AND (B OR D)
 *
 * Luckily this is the goal of this class - to make such conversations.
 */
public class CNF {

    /**
     * This {@code BooleanOperator} enum represents the possible operation that can be placed
     * between two or more variables.
     */
    public enum BooleanOperator {

        /**
         * Indicates that an OR operation should be calculated between 2 or more elements.
         */
        OR,

        /**
         * Indicates that an AND operation should be calculated between 2 or more elements.
         */
        AND
    }

    private ArrayList<ArrayList<String>> items;

    CNF(CNF.Builder builder) {
        if (builder.items != null) {
            items = builder.items;
        }
    }

    /**
     * Get the String representation of the CNF based on the {@link Builder} options set.
     *
     * @return The String representation of CNF, with ',' representing an AND and ';' representing
     * an OR.
     */
    @Nullable
    public String toString() {

        StringBuilder labelsBuilder = new StringBuilder();
        if (items != null && items.size() > 0) {

            boolean hasMoreThan1Entry = (items.size() > 1);
            if (hasMoreThan1Entry) {
                labelsBuilder.append("(");
            }

            for (int index = 0; index < items.size(); index++) {
                labelsBuilder.append(parseCNF(items.get(index)));
                if (index < (items.size() - 1)) {
                    labelsBuilder.append(",");
                }
            }

            if (hasMoreThan1Entry){
                labelsBuilder.append(")");
            }
        }

        if (labelsBuilder.length() == 0){
            return null;
        }
        return labelsBuilder.toString();
    }

    private String parseCNF(ArrayList<String> listOfItems) {

        StringBuilder labelsBuilder = new StringBuilder("(");
        if (listOfItems.size() == 1) {
            labelsBuilder.append(listOfItems.get(0));
            labelsBuilder.append(")");
            return labelsBuilder.toString();
        }

        for (int index2 = 0; index2 < listOfItems.size(); index2++) {
            labelsBuilder.append(listOfItems.get(index2));
            if (index2 != listOfItems.size() - 1) {
                labelsBuilder.append(";");
            }
        }
        labelsBuilder.append(")");
        return labelsBuilder.toString();
    }

    /**
     * The {@code Builder} used to construct a {@link CNF}.
     */
    public static class Builder {

        private ArrayList<ArrayList<String>> items;

        private void initialize() {
            items = new ArrayList<>();
        }

        /**
         * Initialize the {@code Builder} with at least one item. For example if,
         *
         * Builder builder   = new Builder(A)
         * = (A)
         *
         * @param item The original item that is added to the CNF.
         */
        public Builder(@NonNull String item) {
            initialize();
            ArrayList<String> listOfItems = new ArrayList<>();
            listOfItems.add(item);
            items.add(listOfItems);
        }

        /**
         * Initialize the {@code Builder} with a set of items that should be separated with by an OR
         * or an AND. For example if,
         *
         * ArrayList&lt;String&gt; items = new ArrayList&lt;/String&gt;
         * items.add(A)
         * items.add(B)
         * Builder builder   = new Builder(items, BooleanOperator.OR)
         *
         * = (A;B)
         *
         * Builder builder   = new Builder(items, BooleanOperator.AND)
         *
         * = ((A),(B))
         *
         * @param newItems The items that should be added to the CNF.
         * @param operator Indicates the {@link BooleanOperator} which indicates how the
         * {@code newItems} should be related.
         */
        public Builder(@NonNull ArrayList<String> newItems, @NonNull BooleanOperator operator) {
            initialize();

            switch (operator) {
                case AND:
                    for (String item : newItems) {
                        ArrayList<String> listOfItems = new ArrayList<>();
                        listOfItems.add(item);
                        items.add(listOfItems);
                    }
                    break;
                case OR:
                    items.add(newItems);
                    break;
            }
        }

        /**
         * Use the AND operator to add a {@code topic} to the CNF. For example,
         *
         * ArrayList&lt;String&gt; items = new ArrayList&lt;/String&gt;
         * items.add(A)
         * items.add(B)
         * Builder builder   = new Builder(items, BooleanOperator.OR)
         * .and(C)
         *
         * = ((A;B),(C))
         *
         * Builder builder   = new Builder(items, BooleanOperator.AND)
         * .and(C)
         *
         * = ((A),(B),(C))
         *
         * @param topic The item that is added to the CNF.
         */
        public Builder and(@NonNull String topic) {
            ArrayList<String> newTag = new ArrayList<>();
            newTag.add(topic);
            items.add(newTag);
            return this;
        }

        /**
         * Use the AND operator to add the {@code topics} to the CNF. For example,
         *
         * ArrayList&lt;String&gt; itemsAND = new ArrayList&lt;/String&gt;
         * items.add(C)
         * items.add(D)
         *
         * ArrayList&lt;String&gt; items = new ArrayList&lt;/String&gt;
         * items.add(A)
         * items.add(B)
         * Builder builder   = new Builder(items, BooleanOperator.OR)
         * .and(itemsAND, BooleanOperator.AND)
         *
         * = ((A;B),(C),(D))
         *
         * Builder builder   = new Builder(items, BooleanOperator.AND)
         * .and(itemsAND, BooleanOperator.OR)
         *
         * = ((A),(B),(C;D))
         *
         * @param topics The items that should be added to the CNF.
         * @param operator Indicates the {@link BooleanOperator} which indicates how the
         * {@code topics} should be related.
         */
        public Builder and(@NonNull ArrayList<String> topics, @NonNull BooleanOperator operator) {
            switch (operator) {
                case AND:
                    for (String item : topics) {
                        and(item);
                    }
                    break;
                case OR:
                    items.add(topics);
                    break;
            }
            return this;
        }

        /**
         * Use the OR operator to add a {@code topic} to the CNF. For example,
         *
         * ArrayList&lt;String&gt; items = new ArrayList&lt;/String&gt;
         * items.add(A)
         * items.add(B)
         * Builder builder   = new Builder(items, BooleanOperator.OR)
         * .or(C)
         *
         * = ((A;B;C))
         *
         * Builder builder   = new Builder(items, BooleanOperator.AND)
         * .or(C)
         *
         * = ((A;C),(B;C))
         *
         * @param topic The item that is added to the CNF.
         */
        public Builder or(@NonNull String topic) {
            if (items.size() < 1) {
                and(topic);
            } else {
                for (ArrayList<String> listOfTagIds : items) {
                    listOfTagIds.add(topic);
                }
            }
            return this;
        }

        /**
         * Use the OR operator to add the {@code topics} to the CNF. For example,
         *
         * ArrayList&lt;String&gt; itemsOR = new ArrayList&lt;/String&gt;
         * items.add(C)
         * items.add(D)
         *
         * ArrayList&lt;String&gt; items = new ArrayList&lt;/String&gt;
         * items.add(A)
         * items.add(B)
         * Builder builder   = new Builder(items, BooleanOperator.OR)
         * .or(itemsOR, BooleanOperator.AND)
         *
         * = ((A;B;C;D))
         *
         * Builder builder   = new Builder(items, BooleanOperator.AND)
         * .or(itemsOR, BooleanOperator.OR)
         *
         * = ((A;C),(A;D),(B;D),(B;C))
         *
         * @param topics The items that should be added to the CNF.
         * @param operator Indicates the {@link BooleanOperator} which indicates how the
         * {@code topics} should be related.
         */
        public Builder or(@NonNull ArrayList<String> topics, BooleanOperator operator) {
            if (items.size() < 1) {
                and(topics, operator);
            } else {

                switch (operator) {
                    case AND:
                        ArrayList<ArrayList<String>> temp = new ArrayList<>();
                        for (ArrayList<String> tempId : items) {

                            ArrayList<String> tempString = new ArrayList<>();
                            tempString.addAll(tempId);
                            temp.add(tempString);
                        }

                        for (int index = 0; index < temp.size(); index++) {
                            for (int index2 = 1; index2 < topics.size(); index2++) {

                                if (!temp.get(index).contains(topics.get(index2))) {
                                    temp.get(index).add(topics.get(index2));
                                }
                                if (!items.contains(temp.get(index))) {
                                    items.add(temp.get(index));
                                }
                            }
                            if (!items.get(index).contains(topics.get(0))) {
                                items.get(index).add(topics.get(0));
                            }
                        }
                        break;
                    case OR:
                        for (ArrayList<String> listOfTagIds : items) {
                            for (String item : topics) {
                                if (!listOfTagIds.contains(item)) {
                                    listOfTagIds.add(item);
                                }
                            }
                        }
                        break;
                }
            }
            return this;
        }

        /**
         * Return the {@link CNF} of the query.
         *
         * @return The {@link CNF} of the query.
         */
        public CNF build() {
            return new CNF(this);
        }
    }
}
