package com.flybits.commons.library.analytics;

import android.os.Build;
import android.support.annotation.NonNull;

import java.util.Calendar;

/**
 * The {@code AnalyticsBundle} is responsible to define a specific event that is triggered by the
 * SDK or through the application to indicate that a specific action has taken place. This
 * information will be tracked and can view through your instance of the Experience Studio.
 */
public class AnalyticsBundle {

    public static final String PARAM_INTERNAL_FLB_REF   = "timedRef";
    static final String PARAM_OS_TYPE                   = "osType";
    static final String PARAM_OS_VERSION                = "osVersion";
    public static final String PARAM_USER_ID            = "uid";

    private String eventName;
    private boolean isFlybits;
    private Properties appProperties;
    private Properties flbProperties;
    private EventType type;
    private String reference;
    private long timestamp;
    private String userID;

    private AnalyticsBundle(Builder builder){
        this.eventName      = builder.eventName;
        this.appProperties  = builder.appProperties;
        this.isFlybits      = builder.isFlybits;
        this.timestamp      = builder.timestamp;
        this.type           = builder.type;
        this.userID         = builder.userID;
        if (type == EventType.START_TIME_EVENT || type == EventType.END_TIME_EVENT){
            this.reference  = builder.reference;
        }
        flbProperties       = addFlybitsProperties(this.reference);
    }

    private Properties addFlybitsProperties(String reference) {
        Properties properties = new Properties();
        properties.addProperty(PARAM_OS_TYPE, "Android");
        properties.addProperty(PARAM_OS_VERSION, String.valueOf(Build.VERSION.SDK_INT));

        if (reference != null){
            properties.addProperty(PARAM_INTERNAL_FLB_REF, reference);
        }

        if (userID != null) {
            properties.addProperty(PARAM_USER_ID, userID);
        }
        return properties;
    }

    /**
     * Gets the Application's {@link Properties}. These properties are set by the application based
     * on the analytics information it need to collect.
     *
     * @return The application {@link Properties} that are associated to this event.
     */
    public Properties getAppProperties()
    {
        return appProperties;
    }

    /**
     * Gets the event name from this {@link AnalyticsBundle}.
     * @return The event name.
     */
    public String getEventName() {return eventName;}

    /**
     * Gets the Flybits {@link Properties}, these properties are used for internal use by Flybits.
     *
     * @return The Flybits {@link Properties} that are associated to this event.
     */
    public Properties getFlybitsProperties()
    {
        return flbProperties;
    }

    /**
     * Gets a reference object that is unique between an {@link EventType#START_TIME_EVENT} and
     * {@link EventType#END_TIME_EVENT}. This allows Flybits to link the two events together.
     *
     * @return The unique reference of the of the Started/Ended event.
     */
    public String getReference() {return reference;}

    /**
     * Gets the event timestamp from this {@link AnalyticsBundle}.
     * @return The event timestamp.
     */
    public long getTimestamp() {return timestamp;}

    /**
     * Gets the event type from this {@link AnalyticsBundle}.
     * @return The event type.
     */
    public EventType getType() {return type;}

    /**
     * Gets the unique identifier of the user reporting analytics information.
     * @return The device identifier.
     */
    public String getUserID() {return userID;}

    /**
     * Indicates whether or not this event is a Flybits internal event or an event initiated from
     * the application itself.
     *
     * @return true if the event is initiated from Flybits, false otherwise.
     */
    public boolean isFlybits()
    {
        return isFlybits;
    }

    @Override
    public boolean equals(Object o) {

        if (o == null) {
            return false;
        }

        if (o instanceof AnalyticsBundle) {
            AnalyticsBundle thatBundle = (AnalyticsBundle) o;
            if (this.getTimestamp() == thatBundle.getTimestamp()
                    && this.getEventName().equals(thatBundle.getEventName())) {
                return true;
            }
        }

        return false;
    }

    /**
     * The {@code Builder} class used to define a {@link AnalyticsBundle} object using the
     * properties within this {@code Builder}.
     */
    public static final class Builder {

        private String eventName;
        private boolean isFlybits;
        private Properties appProperties;
        private EventType type;
        private String reference;
        private long timestamp;
        private String userID;

        private Builder(){
            this.isFlybits      = false;
            this.appProperties  = new Properties();
            timestamp           = Calendar.getInstance().getTimeInMillis();
            reference           = null;
        }

        /**
         * Constructor used to define a Discrete event, a discrete event is a specific event that
         * occurs within the application or SDK, there is no time associated to discrete events.
         *
         * @param eventName The name of the event that will displayed in the Experience Studio.
         */
        public Builder(String eventName) {
            this();
            this.eventName      = eventName;
            this.type           = EventType.DISCRETE_EVENT;
        }

        /**
         * Constructor used to define a timed event, a timed event is a specific event that
         * associates a duration of time for a specific event within the application or SDK.
         *
         * @param eventName The name of the event that will displayed in the Experience Studio.
         * @param type Indicates whether the event is a {@link EventType#START_TIME_EVENT} or a
         *             {@link EventType#END_TIME_EVENT}.
         * @param reference A unique reference that should be used for link a
         *                  {@link EventType#START_TIME_EVENT} and {@link EventType#END_TIME_EVENT}.
         */
        public Builder(String eventName, @NonNull EventType type, @NonNull String reference) {
            this(eventName);
            this.type           = type;
            if (type == EventType.START_TIME_EVENT || type == EventType.END_TIME_EVENT){
                this.reference  = reference;
            }
        }

        /**
         * Build the {@link AnalyticsBundle} object using the properties that have been set.
         *
         * @return The built {@link AnalyticsBundle} object.
         */
        public AnalyticsBundle build() {
            return new AnalyticsBundle(this);
        }

        /**
         * Sets the a custom timestamp for when the event occurred.
         *
         * @param timestamp The timestamp in milliseconds when this event occurred.
         * @return The {@link Builder} object to which more attributes can be added to.
         */
        public Builder setCustomTimestamp(long timestamp){
            this.timestamp  = timestamp;
            return this;
        }

        /**
         * Indicates that the event an internal SDK event.
         *
         * @return The {@link Builder} object to which more attributes can be added to.
         */
        public Builder setFlybitsEvent(){
            this.isFlybits  = true;
            return this;
        }

        /**
         * Sets the properties - or additional metadata - associated to the event.
         *
         * @param properties The {@link Properties} class which contains all the metadata associated
         *                   to this event.
         * @return The {@link Builder} object to which more attributes can be added to.
         */
        public Builder setAppProperties(@NonNull Properties properties){
            if (properties != null) {
                this.appProperties = properties;
            }
            return this;
        }

        /**
         * Sets unique user identifier for this Analytics request.
         *
         * @param userID The unique user identifier for the Analytics bundle.
         * @return The {@link Builder} object to which more attributes can be added to.
         */
        public Builder setUserID(@NonNull String userID){
            this.userID = userID;
            return this;
        }
    }

    /**
     * This enumerator is used to define all the different events that can be initiated within the
     * using the Analytics component within the Flybits SDK.
     */
    public enum EventType {

        UNKNOWN_EVENT("unknown_event"),
        DISCRETE_EVENT("event_discrete"),
        START_TIME_EVENT("event_timestart"),
        END_TIME_EVENT("event_timeend");

        private final String value;

        /**
         * Constructor that defines the value for each {@code EventType} option.
         *
         * @param value the String value representing each {@code EventType} option.
         */
        EventType(String value) {
            this.value  = value;
        }

        /**
         * Get the String representation for the {@code EventType} option.
         *
         * @return String representation of the {@code EventType} option.
         */
        public String getValue() {
            return this.value;
        }

        /**
         * Get the {@code EventType} enum value corresponding to an String representation.
         *
         * @param value the String representation of the {@code EventType} enum.
         *
         * @return The {@code EventType} enum for the String representation.
         */
        public static EventType fromValue(String value) {
            for(EventType type : EventType.values()) {
                if(type.getValue().equalsIgnoreCase(value)) {
                    return type;
                }
            }
            return UNKNOWN_EVENT;
        }
    }
}
