package com.flybits.commons.library.api.idps;

import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.exceptions.MissingProjectIDException;
import com.flybits.commons.library.logging.Logger;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * The IDP class is the base class for all IDentity Providers (IDP) supported by Flybits. Through
 * this abstract class, developers can define their Identity Providers using the following pieces
 * of information:
 * <ul>
 *     <li>Provider - Which </li>
 *     <li>URL Endpoint - The HTTP(S) route to login/register a user</li>
 *     <li>HTTP Request Body - The POST body of the HTTP(S) request.</li>
 * </ul>
 */
//TODO: Make ProjectID as part of the IDP and deprecate the rest of the ProjectID methods.
public abstract class IDP implements Parcelable{

    private String provider;

    /**
     * Constructor used to define the basic parameters including the {@code provider} and
     * {@code projectID}. These two pieces information are the base for all {@code IDP} classes.
     *
     * @param provider The provider that indicates which IDP this class refers to. Examples can
     *                 include "Facebook", "Twitter", "Google", "Flybits", etc.
     */
    public IDP(String provider){
        this.provider   = provider;
    }

    /**
     * Constructor used to define the basic parameters including the {@code provider} and
     * {@code projectID}. These two pieces information are the base for all {@code IDP} classes.
     *
     * @param in The parcel to un-marshall content associated to the IDP.
     */
    public IDP(Parcel in){
        provider    = in.readString();
    }

    /**
     * The URL Endpoint for the IDP to successfully login/register to.
     *
     * @return The URL Endpoint for login/register APIs.
     */
    public abstract String getAuthenticationEndPoint();

    /**
     * Get the unique identifier for your Flybits project.
     *
     * @param mContext The context of the activity that calling this method.
     * @return The unique identifier for your Flybits project.
     */
    public static String getProjectID(Context mContext) throws MissingProjectIDException{

        String projectId = SharedElements.getProjectID(mContext);
        if (projectId != null && projectId.length()>0){
            return projectId;
        }

        try {
            android.content.pm.ApplicationInfo ai = mContext.getPackageManager().getApplicationInfo(mContext.getPackageName(), PackageManager.GET_META_DATA);
            Bundle bundle = ai.metaData;

            if (bundle.containsKey("com.flybits.projectID")) {
                String projectID    = bundle.getString("com.flybits.projectID");
                SharedElements.setProjectID(mContext, projectID);
                return projectID;
            }
        }catch (PackageManager.NameNotFoundException | NullPointerException e) {
            Logger.exception("IDP.getProjectID", e);
        }
        throw new MissingProjectIDException("Your Flybits ProjectID has not been set. Please make sure you set it within the AndroidManifest.xml file under the meta key: 'com.flybits.projectID'");
    }

    /**
     * Get the provider that indicates which IDP this class refers to. Examples can include
     * "Facebook", "Twitter", "Google", "Flybits", etc.
     *
     * @return The provider that indicates which IDP to connect to.
     */
    public String getProvider(){
        return provider;
    }

    /**
     * Get the HTTP(S) request body String. For this, method each IDP will need to conform to the
     * String that allows this IDP to login/register for the intended user.
     * @param context The current context of the application.
     * @return The JSONObject representation of the HTTP request body that is sent to login/register.
     */
    public JSONObject getRequestBody(Context context)throws MissingProjectIDException{
        return getRequestBody(context, true);
    }

    /**
     * Get the HTTP(S) request body String. For this, method each IDP will need to conform to the
     * String that allows this IDP to login/register for the intended user.
     * @param mContext The current context of the application.
     * @param autoUseProjectId Should the project id in the manifest be activated on login (Can only be false on Flybits IDP Only!)
     * @return The JSONObject representation of the HTTP request body that is sent to login/register.
     */
    public JSONObject getRequestBody(Context mContext, boolean autoUseProjectId)throws MissingProjectIDException{
        JSONObject bodyObject   = getBody();

        if (bodyObject == null){
            bodyObject = new JSONObject();
        }

        if (!autoUseProjectId)
            return bodyObject;

        try {
            bodyObject.put("projectId", getProjectID(mContext));
        }catch (JSONException e){}

        return bodyObject;
    }

    public abstract JSONObject getBody();

    public void writeToParcel(Parcel out, int flags) {
        out.writeString(provider);
    }
}
