package com.flybits.commons.library.analytics;

import android.support.annotation.NonNull;
import android.util.Base64;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * Created by Filip on 12/7/2016.
 */

public class SquishyFormatReader {

    private ByteBuffer mInputBuffer;

    private int version;
    private int flags;

    public SquishyFormatReader(@NonNull String b64Data) throws IOException
    {
        this(Base64.decode(b64Data.getBytes(), Base64.DEFAULT));
    }

    public SquishyFormatReader(@NonNull byte[] data) throws IOException
    {
        mInputBuffer = ByteBuffer.wrap(data);
        mInputBuffer.order(ByteOrder.LITTLE_ENDIAN);

        if (mInputBuffer.getInt() != 0x4D6C6946)
            throw  new IOException("Not Squishy format");

        version = mInputBuffer.getShort();
        flags = mInputBuffer.getShort();

        //Skipped reserve, we are start of data!
        mInputBuffer.position(mInputBuffer.position()+0x8);
    }

    public Object getNext() throws BufferUnderflowException
    {
        int val = mInputBuffer.get();
        int idCode = val & 0xF;
        int numBytes = (val >> 4) & 0xF;

        switch (idCode)
        {
            case 0:
                byte[] data  = new byte[8];
                mInputBuffer.get(data, 0, numBytes);
                return ByteBuffer.wrap(data).order(ByteOrder.LITTLE_ENDIAN).getLong();
            case 1:
                data  = new byte[8];
                mInputBuffer.get(data, 0, numBytes);
                return ByteBuffer.wrap(data).order(ByteOrder.LITTLE_ENDIAN).getDouble();
            case 2:
                return readNullTermedString();
            case 3:
                return true;
            case 4:
                return false;
            case 5:
                return null;
            case 6:
                return readMap();
            case 7:
                return readArray();
        }
        return null;
    }

    private Object[] readArray() {

        int numEntries = mInputBuffer.getInt();

        Object[] array = new Object[numEntries];

        for (int i = 0; i < numEntries; i++)
        {
            array[i] = getNext();
        }

        return array;
    }

    private Map<Object, Object> readMap() {

        Map<Object, Object> map = new LinkedHashMap<>();
        int numEntries = mInputBuffer.getInt();

        for (int i = 0; i < numEntries; i++)
        {
            Object key = getNext();
            Object value = getNext();
            map.put(key, value);
        }

        return map;
    }

    private String readNullTermedString() {

        ByteArrayOutputStream stringBytes = new ByteArrayOutputStream();
        while (true)
        {
            byte b = mInputBuffer.get();
            if (b == 0)
                break;
            stringBytes.write(b);
        }

        return new String(stringBytes.toByteArray());
    }

    public int getVersion()
    {
        return version;
    }

    public int flags()
    {
        return flags;
    }
}
