package com.flybits.commons.library.api.results;


import android.content.Context;
import android.os.Handler;
import android.support.annotation.NonNull;

import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.models.internal.Result;

import java.util.concurrent.ExecutorService;

/**
 * The {@code ObjectResult} class is used to query the server when the response is excepted to an
 * object. If the network request yields an error then null if provided in the {@link #getResult()}
 * method.
 *
 * @param <T> The object that is expected to be returned.
 */
public class ObjectResult<T> extends FlybitsResult {

    private T object;
    private ObjectResultCallback<T> callback;
    private Handler handler;
    private boolean isBackwardsCompat;

    /**
     * Constructor used to initialize the {@code ObjectResult} object.
     *
     * @param context The context of the application.
     * @param callback The {@link ObjectResultCallback} that indicates whether or not the network
     *                 request was successful or not.
     */
    public ObjectResult(@NonNull Context context, ObjectResultCallback<T> callback){
        super(context);
        this.callback   = callback;
    }

    /**
     * Constructor used to initialize the {@code BasicResult} object.
     *
     * @param context The context of the application.
     * @param callback The {@link ObjectResultCallback} that indicates whether or not the network
     *                 request was successful or not.
     * @param service The {@link ObjectResultCallback} that is used to run the network request
     *                thread.
     * @deprecated The more optimized {@link #ObjectResult(Context, ObjectResultCallback, ExecutorService, Handler)}
     * should be used.
     */
    @Deprecated
    public ObjectResult(@NonNull Context context, ObjectResultCallback<T> callback, @NonNull ExecutorService service){
        super(context, service);
        this.callback   = callback;
    }

    /**
     * Constructor used to initialize the {@code BasicResult} object with a Handler object which
     * allows you run on the UI thread.
     *
     * @param context The context of the application.
     * @param callback The {@link ObjectResultCallback} that indicates whether or not the network
     *                 request was successful or not.
     * @param service The {@link ObjectResultCallback} that is used to run the network request
     *                thread.
     * @param handler The handler that allows methods to be run in the UI.
     */
    public ObjectResult(@NonNull Context context, ObjectResultCallback<T> callback, @NonNull ExecutorService service, @NonNull Handler handler){
        super(context, service);
        this.callback       = callback;
        this.handler        = handler;
        isBackwardsCompat   = true;
    }

    /**
     * Get the object that is returned as a result of the network request.
     *
     * @return T The object that is generated from the HTTP response.
     */
    public T getResult(){
        if (getStatus() == QueryStatus.SUCCESS) {
            return object;
        }else{
            return null;
        }
    }
    /**
     * Sets the {@link FlybitsException} that was thrown when request more items to be loaded.
     *
     * @param e The {@link FlybitsException} that caused the network to fail.
     * @return true if the error has been set, false otherwise
     */
    @Override
    public boolean setFailed(final FlybitsException e) {
        if (super.setFailed(e)) {
            if (callback != null){
                if (isBackwardsCompat && handler != null){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.onException(e);
                        }
                    });
                }else if (!isBackwardsCompat){
                    callback.onException(e);
                }
            }
            return true;
        }
        return false;
    }

    /**
     * Sets the result of the network request as successful and returns the object associated
     * to the request.
     *
     * @param result The result that was returned when a successful network request was made.
     */
    public void setSuccess(final @NonNull T result){
        if (setSuccess()) {
            this.object = result;
            if (callback != null){
                if (isBackwardsCompat && handler != null){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.onSuccess(result);
                        }
                    });
                }else if (!isBackwardsCompat){
                    callback.onSuccess(result);
                }
            }
        }
    }

    /**
     * Set the {@link Result} of the network request that contains the object returned from
     * the network request.
     *
     * @param result The {@link Result} of the network request with the result as an object
     *               stored in {@link Result#getResult()}.
     */
    public void setResult(@NonNull Result<T> result){
        if (result.getStatus() == RequestStatus.COMPLETED){
            setSuccess(result.getResult());
        }else{
            setFailed(result.getException());
        }
    }
}
