package com.flybits.commons.library.api.results;

import android.content.Context;

import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ConnectionResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.models.User;
import com.flybits.commons.library.models.internal.Result;

import java.util.concurrent.ExecutorService;

/**
 * The {@code BasicResult} class is used to query the server when no object is expected to be
 * returned, but rather a simple onSuccess or onException with an exception as the reason for the
 * failure.
 */
public class ConnectionResult extends FlybitsResult {

    private ConnectionResultCallback callback;

    /**
     * Constructor used to initialize the {@code BasicResult} object.
     *
     * @param context The context of the application.
     * @param callback The {@link BasicResultCallback} that indicates whether or not the network
     *                 request was successful or not.
     */
    public ConnectionResult(Context context, ConnectionResultCallback callback){
        super(context);
        this.callback   = callback;
    }

    /**
     * Constructor used to initialize the {@code BasicResult} object.
     *
     * @param context The context of the application.
     * @param callback The {@link BasicResultCallback} that indicates whether or not the network
     *                 request was successful or not.
     * @param service The {@link ExecutorService} that is used to run the network request thread.
     */
    public ConnectionResult(Context context, ConnectionResultCallback callback, ExecutorService service){
        super(context, service);
        this.callback   = callback;
    }

    /**
     * Sets the {@link FlybitsException} that was thrown when request more items to be loaded.
     *
     * @param e The {@link FlybitsException} that caused the network to fail.
     * @return true if the error has been set, false otherwise
     */
    @Override
    public boolean setFailed(FlybitsException e) {
        if (super.setFailed(e)) {
            callback.onException(e);
            return true;
        }
        return false;
    }

    /**
     * Sets the result of the network request as successful and it indicates that the user is now
     * connected.
     */
    public void setConnected(){
        if (setSuccess()){
            callback.onConnected();
        }
    }

    /**
     * Sets the result of the network request, and indicates that the user is already connected.
     */
    public void setNotConnected(){
        if (setSuccess()){
            callback.notConnected();
        }
    }

    /**
     * Sets the result of the network request, and indicates that the user is already connected.
     */
    public void setResult(Result result){
        if (result.getStatus() == RequestStatus.COMPLETED){
            setConnected();
        }else{
            setFailed(result.getException());
        }
    }
}
