package com.flybits.commons.library.models.internal;

import android.support.annotation.NonNull;

import java.util.ArrayList;

public class QueryBuilder<T extends QueryBuilder>{

    public long limit;
    public long offset;

    public ArrayList<String> labels;

    public OrderBy orderBy;
    public SortByEnumeratable sortBy;

    public String cachingKey;
    public int cachingLimit;

    /**
     * Default constructor to initializes all variables.
     */
    public QueryBuilder(){
        limit           = 0;
        offset          = -1;
        labels          = new ArrayList<>();
        cachingLimit    = 10;
    }

    /**
     * Default constructor to initializes all variables.
     */
    public QueryBuilder(QueryParameters parameters){
        limit           = parameters.getLimit();
        offset          = parameters.getOffset();
        labels          = parameters.getLabels();
        orderBy         = parameters.getOrderBy();
        sortBy          = parameters.getSortBy();
        cachingKey      = parameters.getCachingKey();
        cachingLimit    = parameters.getCachingLimit();
    }

    /**
     * Sets the caching key for the query. This is useful because it allows you to later retrieve
     * cached copies of the query that you just made.
     *
     * @param cachingKey A unique key that represents the query. If no {@code cachingKey} is set a
     *                   hash of the query parameters will be used.
     * @param limit The maximum number of entities that can be cached.
     */
    protected T setCaching(@NonNull String cachingKey, int limit){
        this.cachingKey     = cachingKey;
        this.cachingLimit   = limit;
        return (T) this;
    }

    /**
     * Adds a constraint to the results, requiring them to have the given labels.
     *
     * @param labels The labels to filter by.
     */
    public T setLabels(String... labels) {
        if (labels != null) {
            this.labels.clear();
            for (String label : labels) {
                if (label != null)
                    this.labels.add(label);
            }
        }
        return (T) this;
    }

    /**
     * Adds a constraint to the results, requiring them to have the given labels.
     *
     * @param labels A list of labels to filter by.
     */
    public T setLabels(@NonNull ArrayList<String> labels) {
        if (labels != null) {
            this.labels.clear();
            this.labels.addAll(labels);
        }
        return (T) this;
    }

    /**
     * Adds a paging mechanism to the request based on the number of returned results wanted and
     * the offset of the next X number of results, where X is a limit.
     *
     * @param limit The maximum number of returned objects.
     * @param offset The offset of where the next X number of response objects will be returned
     *               from where X is the limit.
     */
    public T setPaging(long limit, long offset){
        this.limit  = limit;
        this.offset = offset;
        return (T) this;
    }

    /**
     * Sets the server side sorting, so that the data returned is already presorted by some field.
     * @param sort The field to sort by. Each builder will have a SortBy enum.
     * @param order Either OrderBy.Ascending or OrderBy.Descending.
     */
    protected T setSorting(@NonNull SortByEnumeratable sort, @NonNull OrderBy order) {
        this.sortBy = sort;
        this.orderBy = order;
        return (T) this;
    }
}