package com.flybits.commons.library.models;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;

import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.deserializations.DeserializeUserPref;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.models.internal.Result;

import org.json.JSONObject;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * This {@code UserProperties} class is responsible for getting/creating/clear user preferences
 * associated to the User that is logged into Flybits for a specific project. These preferences are
 * set by the application and the structure can be modified to fit the needs of the application.
 *
 * <p>This class needs to be extended and the {@link #toJSON()} and {@link #fromJSON(String)}
 * need to be implemented by a custom Properties class by the application developer. Once this is
 * done the {@link #get(Context, Class, UserPreferences, ObjectResultCallback)} and
 * {@link #edit(Context, UserPreferences, ObjectResultCallback)} methods can be called to enter the
 * data.</p>
 */
public abstract class UserPreferences {

    static final String API_USER_PREF      = "/kernel/userpref";

    /**
     * This method converts the {@code UserProperties} attributes to a {@code JSONObject} which is
     * called when the {@link #edit(Context, UserPreferences, ObjectResultCallback)} method is
     * initiated by the application.
     *
     * @return The {@code JSONObject} that is generated based on the attributes defined by the
     * application in the custom class that extends the {@link UserPreferences} class.
     */
    public abstract JSONObject toJSON();

    /**
     * This method converts a JSON string into the attributes defined within the custom class that
     * extends the {@link UserPreferences}.
     */
    public abstract void fromJSON(String json);

    /**
     * This is a static method that retrieves a user\'s preferences. If no preferences are set the
     * SDK will initialize the User Preferences for the user based on the {@code defaultsObject}
     * parameter within the method. If a null {@code defaultsObject} parameter and no previous user
     * preferences have been set then a null response will be provided through the
     * {@link ObjectResultCallback#onSuccess(Object)} method.
     *
     * @param context The Context of the application.
     * @param classObject The class object that defines the custom class that extends the
     *                    {@link UserPreferences}.
     * @param defaultsObject This object is a default in the event that no User Preferences have
     *                       been previously set.
     * @param callback The callback used to indicate whether or not the disconnection was successful
     *                 or not.
     * @return The {@link ObjectResult} is an object that is returned from the Server after a HTTP
     * request is made.
     */
    public static ObjectResult<UserPreferences> get(@NonNull final Context context, @NonNull final Class classObject,
                                                    final UserPreferences defaultsObject, ObjectResultCallback<UserPreferences> callback){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<UserPreferences> request = new ObjectResult<>(context, callback, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {

                DeserializeUserPref deserializer    = (defaultsObject != null) ? new DeserializeUserPref<>(defaultsObject.getClass())
                        : new DeserializeUserPref<>(classObject);

                String methodIdentifier             = "Self.get";
                try {
                    final Result<UserPreferences> getUserPrefs = FlyAway.get(context, API_USER_PREF, deserializer, methodIdentifier, UserPreferences.class);
                    if (getUserPrefs.getStatus() == RequestStatus.NOT_FOUND && defaultsObject != null){

                        String jsonBody     = deserializer.toJson(defaultsObject).toString();
                        final Result<UserPreferences> createUserPrefs = FlyAway.post(context, API_USER_PREF, jsonBody, deserializer, methodIdentifier, UserPreferences.class);
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                request.setResult(createUserPrefs);
                            }
                        });
                    }else if (getUserPrefs.getStatus() == RequestStatus.NOT_FOUND){

                        final Result<UserPreferences> result = new Result(200, "");
                        result.setResponse(null);
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                request.setResult(result);
                            }
                        });

                    }else {
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                request.setResult(getUserPrefs);
                            }
                        });
                    }

                }catch (final FlybitsException e){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setFailed(e);
                        }
                    });
                }
            }
        });
        return request;
    }

    /**
     * This method is useful when you want to update the user preferences of the logged in
     * {@link User}. The {@code object} parameter cannot be null and it overrides all the
     * preferences previously saved.
     *
     * @param context The Context of the application.
     * @param object The object that should be used to save user preferences with.
     * @param callback The callback used to indicate whether or not the disconnection was successful
     *                 or not.
     * @return The {@link ObjectResult} is an object that is returned from the Server after a HTTP
     * request is made.
     */
    public static ObjectResult<UserPreferences> edit(@NonNull final Context context, @NonNull final UserPreferences object,
                                                     ObjectResultCallback<UserPreferences> callback){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<UserPreferences> request = new ObjectResult<>(context, callback, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {

                DeserializeUserPref deserializer    = new DeserializeUserPref<>(object.getClass());
                String methodIdentifier             = "Self.edit";
                String body                         = deserializer.toJson(object).toString();

                try {
                    final Result<UserPreferences> putUserPrefs = FlyAway.put(context, API_USER_PREF, body,
                            deserializer, methodIdentifier, UserPreferences.class);

                    if (putUserPrefs.getStatus() == RequestStatus.NOT_FOUND){

                        String jsonBody     = deserializer.toJson(object).toString();
                        final Result<UserPreferences> createUserPrefs = FlyAway.post(context, API_USER_PREF, jsonBody, deserializer, methodIdentifier, UserPreferences.class);
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                request.setResult(createUserPrefs);
                            }
                        });
                    }else {
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                request.setResult(putUserPrefs);
                            }
                        });
                    }

                }catch (final FlybitsException e){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setFailed(e);
                        }
                    });
                }
            }
        });
        return request;
    }

    /**
     * Delete all the user preferences associated to the logged in {@link User}.
     *
     * @param context The Context of the application.
     * @param callback The callback used to indicate whether or not the disconnection was successful
     *                 or not.
     * @return The {@link ObjectResult} is an object that is returned from the Server after a HTTP
     * request is made.
     */
    public static BasicResult clear(@NonNull final Context context, BasicResultCallback callback){
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(context, callback, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                String methodIdentifier             = "Self.clear";
                try {
                    final Result<UserPreferences> deleteUserPref = FlyAway.delete(context, API_USER_PREF, methodIdentifier, null);
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setResult(deleteUserPref);
                        }
                    });

                }catch (final FlybitsException e){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setFailed(e);
                        }
                    });
                }
            }
        });
        return request;
    }
}