package com.flybits.commons.library;

import android.content.Context;
import android.content.SharedPreferences;
import android.support.annotation.NonNull;

import com.flybits.commons.library.api.FlybitsManager;
import com.flybits.commons.library.api.results.callbacks.ConnectionResultCallback;
import com.flybits.commons.library.models.User;
import com.flybits.commons.library.utils.Utilities;
import com.flybits.internal.db.CommonsDatabase;

import java.util.ArrayList;

import static android.content.Context.MODE_PRIVATE;

/**
 * This class is used to get and set all variables that can be shared across the various Flybits'
 * Android SDKs. Currently the items that can be shared between the SDKs are as follows;
 * <ul>
 *     <li>Language Information</li>
 *     <li>Device Identifier</li>
 *     <li>User Identifier</li>
 *     <li>JWT</li>
 *     <li>Project ID</li>
 *     <li>Connected IDP</li>
 * </ul>
 */
public class SharedElements {

    public static final String PREF_LANGUAGE_CODES      = "com.flybits.language.codes";
    public static final String PREF_JWT_TOKEN           = "com.flybits.jwt.token";
    public static final String PREF_IDP_CONNECTED       = "com.flybits.idp.connected";
    public static final String PREF_PROJECT_ID          = "com.flybits.project.id";

    /**
     * Get the IDP that the application used to connect to Flybits with. If the application is not
     * connected to Flybits then "" empty string will be returned.
     *
     * @param context The context of the application.
     * @return The saved string representation of the IDP used to connect to Flybits with, or "" if
     * the user is not connected to Flybits.
     */
    public static String getConnectedIDP(Context context){
        SharedPreferences preferences   = getPreferences(context);
        return preferences.getString(PREF_IDP_CONNECTED, "");
    }

    /**
     * Get the previously saved {@link com.flybits.commons.library.models.User#deviceID}.
     *
     * @param context The context of the application.
     * @return The saved {@link com.flybits.commons.library.models.User#deviceID} or "" if no
     * {@link com.flybits.commons.library.models.User#deviceID} is saved.
     */
    public static String getDeviceID(Context context){
        User user = CommonsDatabase.getDatabase(context).userDao().getSingle();
        if (user != null){
            return user.getDeviceID();
        }
        return "";
    }

    /**
     * Get the ArrayList representation of the language codes set for this application.
     *
     * @param context The context of the application.
     * @return The ArrayList representation of the language codes set for this application.
     */
    public static ArrayList<String> getEnabledLanguagesAsArray(Context context){
        SharedPreferences preferences = getPreferences(context);
        String languageCodes = preferences.getString(PREF_LANGUAGE_CODES, "");
        return Utilities.convertLocalizationStringToList(languageCodes);
    }

    /**
     * Get the String representation of the language codes set for this application.
     *
     * @param context The context of the application.
     * @return The String representation of the language codes set for this application or "" if no
     * language is set.
     */
    public static String getEnabledLanguagesAsString(Context context){
        SharedPreferences preferences = getPreferences(context);
        return preferences.getString(PREF_LANGUAGE_CODES, "");
    }

    /**
     * Get the unique Flybits Project Identifier associated to your project. This project is set
     * through the {@link com.flybits.commons.library.api.idps.IDP#getProjectID(Context)}.
     *
     * @param context The context of the application.
     * @return The saved unique Project identifier which can be retrieved from the
     * <a href="https://developer.flybits.com">Developer Portal</a>  or "" if no project id.
     */
    public static String getProjectID(Context context){
        SharedPreferences preferences   = getPreferences(context);
        return preferences.getString(PREF_PROJECT_ID, "");
    }

    /**
     * Get the previously saved {@code JWT} which is obtained when calling
     * {@link FlybitsManager#connect(ConnectionResultCallback)} the first time.
     *
     * @param context The context of the application.
     * @return The saved {@code JWT} which is obtained once the application logs into Flybits  or ""
     * if the application has not successfully received a {@code JWT} token.
     */
    public static String getSavedJWTToken(Context context){
        SharedPreferences preferences   = getPreferences(context);
        return preferences.getString(PREF_JWT_TOKEN, "");
    }

    /**
     * Get the previously saved {@link com.flybits.commons.library.models.User#id}.
     *
     * @param context The context of the application.
     * @return The saved {@link com.flybits.commons.library.models.User#id} or "" if no
     * {@link com.flybits.commons.library.models.User#id} is saved.
     */
    public static String getUserID(Context context){
        User user = CommonsDatabase.getDatabase(context).userDao().getSingle();
        if (user != null){
            return user.getId();
        }
        return "";
    }

    /**
     * Sets the IDP that was used to connect to Flybits with such as Flybits, Anonymous, OAuth, etc.
     *
     * @param context The context of the application.
     * @param idp The string representation of the IDP used to connect to Flybits with.
     */
    public static void setConnectedIDP(@NonNull Context context, @NonNull String idp) {
        setVariable(context, PREF_IDP_CONNECTED, idp);
    }

    /**
     * Sets the unique JWT Token obtained from the Flybits Core for the user/device combination.
     *
     * @param context The context of the application.
     * @param jwtToken The unique JWT Token that is used by Flybits to identify a user/device
     *                 combination.
     */
    public static void setJWTToken(Context context, @NonNull String jwtToken) {
        setVariable(context, PREF_JWT_TOKEN, jwtToken);
    }

    /**
     * Sets the localization values of the device.
     *
     * @param context The context of the application.
     * @param listOfLanguages The array of languages that should be used for this device.
     */
    public static void setLocalization(Context context, ArrayList<String> listOfLanguages) {
        setVariable(context, PREF_LANGUAGE_CODES, Utilities.convertLocalizationCodeToString(listOfLanguages));
    }

    /**
     * Sets the Flybits Project Identifier which can be used by other components/sdks within the
     * Flybits ecosystem.
     *
     * @param context The context of the application.
     * @param projectID The unique Flybits Project Identifier that represents this project's
     *                  application.
     * @deprecated Use {@link FlybitsManager.Builder#setProjectId(String)} instead as it is a more
     * optimized.
     */
    public static void setProjectID(Context context, String projectID){
        setVariable(context, PREF_PROJECT_ID, projectID);
    }

    /**
     * Get the default shared preferences for the application.
     *
     * @param context The context of the application.
     * @return The default SharedPreferences object for the application.
     */
    public static SharedPreferences getPreferences(Context context){
        return context.getSharedPreferences("FLYBITS_PREF", MODE_PRIVATE );
    }

    private static void setVariable(Context context, @NonNull String key, @NonNull String value){
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(key, value);
        editor.apply();
    }
}
