package com.flybits.commons.library.analytics.storage.sql;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.util.Log;

import com.flybits.commons.library.analytics.AnalyticsBundle;
import com.flybits.commons.library.analytics.Properties;
import com.flybits.commons.library.logging.Logger;

import org.json.JSONException;

import static com.flybits.commons.library.analytics.AnalyticsBundle.PARAM_INTERNAL_FLB_REF;

class AnalyticsQueueStorageHelper extends SQLiteOpenHelper {

    private static final int DATABASE_VERSION = 3;
    private static AnalyticsQueueStorageHelper mInstance;

    private AnalyticsQueueStorageHelper(Context context) {
        super(context, "flybits_analytics_db", null, DATABASE_VERSION);
    }

    @Override
    public void onCreate(SQLiteDatabase db) {
        db.execSQL(AnalyticsContract.CREATE_TABLE);

    }

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        db.execSQL(AnalyticsContract.DELETE_TABLE);
        onCreate(db);
    }

    public static synchronized AnalyticsQueueStorageHelper getInstance(Context context) {
        if (mInstance == null) {
            mInstance = new AnalyticsQueueStorageHelper(context);
        }
        return mInstance;
    }

    public void add(SQLiteDatabase db, AnalyticsBundle bundle) {
        if (bundle == null) {
            return;
        }

        ContentValues values = new ContentValues();
        values.put(AnalyticsContract.AnalyticsEntry.COLUMN_NAME, bundle.getEventName());
        values.put(AnalyticsContract.AnalyticsEntry.COLUMN_TIMESTAMP, bundle.getTimestamp());
        values.put(AnalyticsContract.AnalyticsEntry.COLUMN_TYPE, bundle.getType().getValue());
        values.put(AnalyticsContract.AnalyticsEntry.COLUMN_FLYBITS, bundle.isFlybits() ? 1 : 0);
        values.put(AnalyticsContract.AnalyticsEntry.COLUMN_USER_ID, bundle.getUserID());

        try {
            values.put(AnalyticsContract.AnalyticsEntry.COLUMN_PROPERTIES, bundle.getAppProperties().toJSON());
        } catch (JSONException e) {
            Logger.exception("AnalyticsQueueStorageHelper.add", e);
        }

        try {
            values.put(AnalyticsContract.AnalyticsEntry.COLUMN_FLB_PROPERTIES, bundle.getFlybitsProperties().toJSON());
        } catch (JSONException e) {
            Logger.exception("AnalyticsQueueStorageHelper.add", e);
        }

        db.insertWithOnConflict(AnalyticsContract.AnalyticsEntry.TABLE_NAME, null, values, SQLiteDatabase.CONFLICT_REPLACE);
    }

    public AnalyticsBundle getNext(SQLiteDatabase db) {

        Cursor cursor = db.rawQuery("SELECT * FROM " + AnalyticsContract.AnalyticsEntry.TABLE_NAME + " ORDER BY timestamp ASC LIMIT 1", new String[0]);
        cursor.moveToFirst();

        if (cursor.getCount() == 0)
            return null;

        String name             = cursor.getString(cursor.getColumnIndex(AnalyticsContract.AnalyticsEntry.COLUMN_NAME));
        long timestamp          = cursor.getLong(cursor.getColumnIndex(AnalyticsContract.AnalyticsEntry.COLUMN_TIMESTAMP));
        String properties       = cursor.getString(cursor.getColumnIndex(AnalyticsContract.AnalyticsEntry.COLUMN_PROPERTIES));
        String flbProperties    = cursor.getString(cursor.getColumnIndex(AnalyticsContract.AnalyticsEntry.COLUMN_FLB_PROPERTIES));
        boolean isFlybits       = cursor.getInt(cursor.getColumnIndex(AnalyticsContract.AnalyticsEntry.COLUMN_FLYBITS)) == 1;
        String userID           = cursor.getString(cursor.getColumnIndex(AnalyticsContract.AnalyticsEntry.COLUMN_USER_ID));
        AnalyticsBundle.EventType type = AnalyticsBundle.EventType.fromValue(cursor.getString(cursor.getColumnIndex(AnalyticsContract.AnalyticsEntry.COLUMN_TYPE)));
        cursor.close();

        try {
            if (type == AnalyticsBundle.EventType.DISCRETE_EVENT){
                AnalyticsBundle.Builder bundle = new AnalyticsBundle.Builder(name)
                        .setAppProperties(new Properties(properties))
                        .setUserID(userID)
                        .setCustomTimestamp(timestamp);

                if (isFlybits){
                    bundle.setFlybitsEvent();
                }
                return bundle.build();
            }else if(type == AnalyticsBundle.EventType.START_TIME_EVENT || type == AnalyticsBundle.EventType.END_TIME_EVENT){

                Properties flbPropertiesObj = new Properties(flbProperties);
                if (flbPropertiesObj.getMap().containsKey(PARAM_INTERNAL_FLB_REF)) {

                    AnalyticsBundle.Builder bundle = new AnalyticsBundle.Builder(name, type, flbPropertiesObj.getMap().get(PARAM_INTERNAL_FLB_REF).toString())
                            .setAppProperties(new Properties(properties))
                            .setUserID(userID)
                            .setCustomTimestamp(timestamp);

                    if (isFlybits){
                        bundle.setFlybitsEvent();
                    }
                    return bundle.build();
                }
            }
            return null;
        } catch (JSONException e) {
            Logger.exception("AnalyticsQueueStorageHelper.getNext", e);
        }

        return null;
    }

    public AnalyticsBundle[] getAll(SQLiteDatabase db) {

        Cursor cursor = db.rawQuery("SELECT * FROM " + AnalyticsContract.AnalyticsEntry.TABLE_NAME, new String[0]);
        AnalyticsBundle[] bundleList = new AnalyticsBundle[cursor.getCount()];
        int i = 0;

        if (cursor.moveToFirst()) {
            do {
                String name             = cursor.getString(cursor.getColumnIndex(AnalyticsContract.AnalyticsEntry.COLUMN_NAME));
                long timestamp          = cursor.getLong(cursor.getColumnIndex(AnalyticsContract.AnalyticsEntry.COLUMN_TIMESTAMP));
                String properties       = cursor.getString(cursor.getColumnIndex(AnalyticsContract.AnalyticsEntry.COLUMN_PROPERTIES));
                String flbProperties    = cursor.getString(cursor.getColumnIndex(AnalyticsContract.AnalyticsEntry.COLUMN_FLB_PROPERTIES));
                boolean isFlybits       = cursor.getInt(cursor.getColumnIndex(AnalyticsContract.AnalyticsEntry.COLUMN_FLYBITS)) == 1;
                String userID           = cursor.getString(cursor.getColumnIndex(AnalyticsContract.AnalyticsEntry.COLUMN_USER_ID));
                AnalyticsBundle.EventType type = AnalyticsBundle.EventType.fromValue(cursor.getString(cursor.getColumnIndex(AnalyticsContract.AnalyticsEntry.COLUMN_TYPE)));

                Log.d("Testing", "isFlybitsL " + isFlybits);

                try {

                    AnalyticsBundle.Builder bundle = null;
                    if (type == AnalyticsBundle.EventType.DISCRETE_EVENT){
                        bundle = new AnalyticsBundle.Builder(name);
                    }else if(type == AnalyticsBundle.EventType.START_TIME_EVENT || type == AnalyticsBundle.EventType.END_TIME_EVENT){

                        Properties flbPropertiesObj = new Properties(flbProperties);
                        if (flbPropertiesObj.getMap().containsKey(PARAM_INTERNAL_FLB_REF)) {
                            bundle = new AnalyticsBundle.Builder(name, type, flbPropertiesObj.getMap().get(PARAM_INTERNAL_FLB_REF).toString());
                        }
                    }

                    if (bundle != null) {
                        bundle.setAppProperties(new Properties(properties))
                                .setCustomTimestamp(timestamp);

                        if (isFlybits){
                            bundle.setFlybitsEvent();
                        }

                        if (userID != null){
                            bundle.setUserID(userID);
                        }

                        bundleList[i++] = bundle.build();
                    }
                } catch (JSONException e) {
                    Logger.exception("AnalyticsQueueStorageHelper.getAll", e);
                }
            } while (cursor.moveToNext());
        }

        cursor.close();
        return bundleList;
    }

    public int getSize(SQLiteDatabase db) {

        Cursor cursor = db.rawQuery("SELECT * FROM " + AnalyticsContract.AnalyticsEntry.TABLE_NAME, new String[0]);
        int size = cursor.getCount();
        cursor.close();
        return size;
    }

    public void remove(SQLiteDatabase db, int count) {
        if (count != 0)
            db.delete(AnalyticsContract.AnalyticsEntry.TABLE_NAME,
                    String.format("%1$s in (SELECT %1$s from "+ AnalyticsContract.AnalyticsEntry.TABLE_NAME+" order by %1$s desc limit ?)",
                            AnalyticsContract.AnalyticsEntry.COLUMN_TIMESTAMP), new String[]{String.valueOf(count)});
    }

}
