package com.flybits.commons.library.analytics.converter;

import com.flybits.commons.library.analytics.AnalyticsBundle;
import com.flybits.commons.library.analytics.Properties;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import static com.flybits.commons.library.analytics.AnalyticsBundle.PARAM_INTERNAL_FLB_REF;
import static com.flybits.commons.library.analytics.AnalyticsBundle.PARAM_USER_ID;

public class JsonConverter implements Converter<String> {

    private final String ATTRIBUTE_NAME             = "name";
    private final String ATTRIBUTE_LOGGED_AT        = "loggedAt";
    private final String ATTRIBUTE_TYPE             = "type";
    private final String ATTRIBUTE_APP_PROPERTIES   = "properties";
    private final String ATTRIBUTE_FLB_PROPERTIES   = "flbProperties";
    private final String ATTRIBUTE_IS_FLB           = "isFlybits";

    public JsonConverter(){}

    @Override
    public String to(AnalyticsBundle[] bundles) throws JSONException {

        JSONArray rootArray = new JSONArray();

        for (AnalyticsBundle bundle : bundles)
        {
            JSONObject bundleObj = new JSONObject();
            Properties properties = bundle.getAppProperties();
            JSONObject propertyObject = new JSONObject(properties.toJSON());

            Properties flbProperties = bundle.getFlybitsProperties();
            JSONObject flbpropertyObject = new JSONObject(flbProperties.toJSON());

            bundleObj.put(ATTRIBUTE_NAME, bundle.getEventName());
            bundleObj.put(ATTRIBUTE_LOGGED_AT, bundle.getTimestamp());
            bundleObj.put(ATTRIBUTE_TYPE, bundle.getType().getValue());
            bundleObj.put(ATTRIBUTE_APP_PROPERTIES, propertyObject);
            bundleObj.put(ATTRIBUTE_FLB_PROPERTIES, flbpropertyObject);
            bundleObj.put(ATTRIBUTE_IS_FLB, bundle.isFlybits());
            rootArray.put(bundleObj);
        }

        return rootArray.toString();
    }

    @Override
    public AnalyticsBundle[] from(String data) throws JSONException {

        JSONArray rootArray = new JSONArray(data);

        int size = rootArray.length();

        AnalyticsBundle[] bundles = new AnalyticsBundle[size];

        for (int i = 0; i < size; i++)
        {
            JSONObject bundleObj = rootArray.getJSONObject(i);

            if (bundleObj.has(ATTRIBUTE_NAME) && bundleObj.has(ATTRIBUTE_LOGGED_AT) && bundleObj.has(ATTRIBUTE_TYPE)
                    && bundleObj.has(ATTRIBUTE_APP_PROPERTIES)) {

                String name              = bundleObj.getString(ATTRIBUTE_NAME);
                long timestamp           = bundleObj.getLong(ATTRIBUTE_LOGGED_AT);
                boolean isFlybits        = bundleObj.getBoolean(ATTRIBUTE_IS_FLB);
                String typeString        = bundleObj.getString(ATTRIBUTE_TYPE);

                Properties properties    = new Properties();
                if (bundleObj.has(ATTRIBUTE_FLB_PROPERTIES)) {
                    properties = new Properties(bundleObj.getJSONObject(ATTRIBUTE_APP_PROPERTIES).toString());
                }

                Properties flbProperties    = new Properties();
                if (bundleObj.has(ATTRIBUTE_FLB_PROPERTIES)) {
                    flbProperties = new Properties(bundleObj.getJSONObject(ATTRIBUTE_FLB_PROPERTIES).toString());
                }
                AnalyticsBundle.EventType type  = AnalyticsBundle.EventType.fromValue(typeString);

                AnalyticsBundle.Builder event;
                if (type == AnalyticsBundle.EventType.DISCRETE_EVENT) {
                    event = new AnalyticsBundle.Builder(name);
                    if (flbProperties.getMap().containsKey(PARAM_USER_ID)){
                        event.setUserID((String) flbProperties.getMap().get(PARAM_USER_ID));
                    }

                }else if ((type == AnalyticsBundle.EventType.END_TIME_EVENT || type == AnalyticsBundle.EventType.START_TIME_EVENT )
                        && flbProperties.getMap().containsKey(PARAM_INTERNAL_FLB_REF)){

                    event = new AnalyticsBundle.Builder(name, type, flbProperties.getMap().get(PARAM_INTERNAL_FLB_REF).toString());
                    if (flbProperties.getMap().containsKey(PARAM_USER_ID)){
                        event.setUserID((String) flbProperties.getMap().get(PARAM_USER_ID));
                    }

                }else{
                    break;
                }

                event.setCustomTimestamp(timestamp).setAppProperties(properties);

                if (isFlybits){
                    event.setFlybitsEvent();
                }
                bundles[i] = event.build();
            }
        }

        return bundles;
    }

}
