package com.flybits.commons.library.api;

import android.content.Context;

import com.flybits.commons.library.models.User;

/**
 * The {@code FlybitsScope} class is an abstract class that is used to define an SDK scope which is
 * used to define the properties of an SDK. For example, the Flybits Context SDK would extend this
 * {@code FlybitsScope} class which will then be used when defining the
 * {@link com.flybits.commons.library.api.FlybitsManager.Builder#addScope(FlybitsScope)}.
 */
public abstract class FlybitsScope {

    private String scopeName;

    /**
     * This constructor is used to define this {@link FlybitsScope} based on a unique Scope code.
     *
     * @param name The unique name of the Scope.
     */
    public FlybitsScope(String name) {
        this.scopeName = name;
    }

    /**
     * Get the unique name of the Scope.
     *
     * @return The unique name of the Scope.
     */
    public String getScopeName() {
        return scopeName;
    }

    /**
     * Indicates that the login was successfully made and that the appropriate actions for an SDK
     * should be performed.
     *
     * @param context The current Context of the application
     * @param user The logged in {@link User}.
     */
    public abstract void onConnected(Context context, User user);

    /**
     * Indicates that the logout was successful. The SDK will need to make the appropriate actions.
     *
     * @param context The current Context of the application.
     * @param jwtToken The JWT token associated to the login user.
     */
    public abstract void onDisconnected(Context context, String jwtToken);

    /**
     * Indicates the logged in {@link User} has requested their account to be disabled. Once
     * successful then the SDK will need to perform the appropriate action.
     *
     * @param context The current Context of the application
     * @param jwtToken The JWT token associated to the login user.
     */
    public abstract void onAccountDestroyed(Context context, String jwtToken);
}