package com.flybits.commons.library.analytics;

import android.support.annotation.NonNull;
import android.util.Base64;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Map;

/**
 * Created by Filip on 12/1/2016.
 */

public class SquishyFormat {

    /*
    Stores key/value pairs in a binary format. 0 bytes are stripped out to save space.

    ===Header===

    +-----------+---+---+---+---------+---+-------+---+----------+---+---+---+---+---+---+---+
    |     0     | 1 | 2 | 3 |    4    | 5 |   6   | 7 |    8     | 9 | A | B | C | D | E | F |
    +-----------+---+---+---+---------+---+-------+---+----------+---+---+---+---+---+---+---+
    |        Signature      |    Version  |   Flags   |              Reserved                |
    +-----------+---+---+---+---------+---+-------+---+----------+---+---+---+---+---+---+---+

    Signature: Unique id of this file (check if this is here)
    Version: Current version
    Flags:
        0: Byte Stripping
        1: Compressed
    Reserved: Space for extra features

    ===Data===

    All data is stored by a single byte, followed by it's entry. This byte is split into a lo and hi
    value. The lo byte is the type ID and the high byte is the number of bytes to read if byte
    stripping enabled (not-applicable to maps and strings).

    Type IDs:
    0: Number
    1: Floating Number
    2: String (Null terminated)
    3: Boolean (True)
    4: Boolean (False)
    5: Null
    6: Key/Val Map
    7: Array

    ===Map Datatypes===

    Following the datatype byte, a "number of entries" int32 follows. This means there is a max
    of 65536 entries in a map. The contents of the map follows, which is organized in the same
    way as the other data.

    EG:

    0x00, 0x5       <- Number
    0x02, "TEST"    <- String
    0x06, 0x0002    <- Map start, 2 entries
    0x02, "Key1",   <- First entry's key
    0x02, "Value1", <- First entry's value
    0x02, "Key2",   <- Second entry's key
    0x02, "Value2", <- Second entry's value
    0x02, "TEST2"   <- Another string not in map

    ==Array Datatypes==

    Like maps, except only one value. After the datatype byte, a *number of entries* int32 follows.

     */

    public final static int DATATYPE_NUMBER = 0;
    public final static int DATATYPE_FLOATNUMBER = 1;
    public final static int DATATYPE_STRING = 2;
    public final static int DATATYPE_BOOLEAN_FALSE = 4;
    public final static int DATATYPE_BOOLEAN_TRUE = 3;
    public final static int DATATYPE_NULL = 5;
    public final static int DATATYPE_MAP = 6;
    public final static int DATATYPE_ARRAY = 7;

    public final static int FLAG_BYTE_STRIPPING = 0;

    private ByteArrayOutputStream mOutputByteArray;

    private boolean mIsByteStripping = false;

    private SquishyFormat() throws IOException
    {
        this(32);
    }

    private SquishyFormat(int size) throws IOException {
        size+=0x10;
        mOutputByteArray = new ByteArrayOutputStream(size);
        mIsByteStripping = true;

        //Signature
        mOutputByteArray.write(new byte[]{'F', 'i', 'l', 'M'}, 0, 4);
        //Version
        mOutputByteArray.write(new byte[]{0x01, 0x00});
        //Flags
        short flags = 0;
        flags |= (mIsByteStripping ? 1 : 0) << FLAG_BYTE_STRIPPING;
        mOutputByteArray.write(ByteBuffer.allocate(2).order(ByteOrder.LITTLE_ENDIAN).putShort(flags).array());

        //Skip to beginning of data
        for (int i = 0; i < 8; i++)
            mOutputByteArray.write(0);
    }

    public static SquishyFormat begin() throws IOException
    {
        return new SquishyFormat();
    }

    public static SquishyFormat begin(int size) throws IOException
    {
        return new SquishyFormat(size);
    }

    public byte[] end()
    {
        return mOutputByteArray.toByteArray();
    }

    public String endAsBase64()
    {
        byte[] encoded = Base64.encode(mOutputByteArray.toByteArray(), Base64.DEFAULT);
        return new String(encoded);
    }

    public void writeNumber(long value) throws IOException
    {
        byte[] bytes = ByteBuffer.allocate(8).order(ByteOrder.LITTLE_ENDIAN).putLong(value).array();
        int numBytes = countNonZeroBytes(bytes);

        mOutputByteArray.write((numBytes << 4) | DATATYPE_NUMBER);

        if (mIsByteStripping)
            mOutputByteArray.write(bytes, 0, numBytes);
        else
            mOutputByteArray.write(bytes);
    }

    public void writeFNumber(double value) throws IOException
    {
        byte[] bytes = ByteBuffer.allocate(8).order(ByteOrder.LITTLE_ENDIAN).putDouble(value).array();
        int numBytes = countNonZeroBytes(bytes);

        mOutputByteArray.write((numBytes << 4) | DATATYPE_FLOATNUMBER);

        if (mIsByteStripping)
            mOutputByteArray.write(bytes, 0, numBytes);
        else
            mOutputByteArray.write(bytes);
    }

    public void writeString(@NonNull String value) throws IOException
    {
        byte[] bytes = value.getBytes();
        mOutputByteArray.write(DATATYPE_STRING);
        mOutputByteArray.write(bytes);
        mOutputByteArray.write(0);
    }

    public void writeBoolean(boolean value) throws IOException
    {
        mOutputByteArray.write(value ? DATATYPE_BOOLEAN_TRUE : DATATYPE_BOOLEAN_FALSE);
    }

    public void writeNull() throws IOException
    {
        mOutputByteArray.write(DATATYPE_NULL);
    }

    public void writeArray(@NonNull Object[] array) throws IOException
    {
        mOutputByteArray.write(DATATYPE_ARRAY);
        mOutputByteArray.write(ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN).putInt(array.length).array());

        for (int i = 0; i < array.length; i++)
            write(array[i]);
    }

    public void writeMap(@NonNull Map map) throws IOException {
        mOutputByteArray.write(DATATYPE_MAP);
        mOutputByteArray.write(ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN).putInt(map.size()).array());

        for (Object key : map.keySet())
        {
            write(key);
            write(map.get(key));
        }
    }

    private void write(Object o) throws IOException {
        if (o instanceof String)
        {
            writeString((String) o);
        }
        else if (o instanceof Integer)
        {
            writeNumber((Integer) o);
        }
        else if (o instanceof Long)
        {
            writeNumber((Long) o);
        }
        else if (o instanceof Boolean)
        {
            writeBoolean((Boolean)o);
        }
        else
            writeString(o.toString());
    }

    private int countNonZeroBytes(byte[] value)
    {
        int numZeros = 0;
        for (int i = 0; i < value.length; i++)
        {
            if (value[i] == 0)
                numZeros++;
            else
                numZeros = 0;
        }
        return value.length - numZeros;
    }

}
