package com.flybits.commons.library.models.internal;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

/**
 * The {@link QueryParameters} class is used to define all common parameters that can be added to
 * any GET request in order to filter the results requested. For example, you may want to add paging
 * to your GET request in order to get a different set of results.
 */
public class QueryParameters {

    private long limit;
    private long offset;
    private ArrayList<String> labels;
    private OrderBy orderBy = null;
    private SortByEnumeratable sortBy = null;
    private String cachingKey;

    /**
     * Default constructor that initializes all the local variables including {@code limit} and
     * {@code offset} used for paging, and an empty {@code labels} list for filtering by label.
     */
    public QueryParameters(QueryBuilder builder) {
        limit       = builder.limit;
        offset      = builder.offset;
        labels      = builder.labels;
        sortBy      = builder.sortBy;
        orderBy     = builder.orderBy;
        cachingKey  = builder.cachingKey;
    }

    /**
     * Get the list of query parameters that should be added to the GET request.
     *
     * @return A list of parameters that should be added to the GET query.
     */
    public Map<String, ArrayList<String>> getQueryParams(){

        Map<String, ArrayList<String>> params = new HashMap<>();

        if (limit > 0 && offset > -1) {

            ArrayList<String> limitParam = new ArrayList<>();
            limitParam.add( String.valueOf(limit));
            params.put("limit", limitParam);

            ArrayList<String> offsetParam = new ArrayList<>();
            offsetParam.add(String.valueOf(offset));
            params.put("offset", offsetParam);
        }

        if (labels.size() > 0) {
            ArrayList<String> labelsParam = new ArrayList<>();
            labelsParam.addAll(labels);
            params.put("labels", labelsParam);
        }

        if (sortBy != null && orderBy != null) {
            ArrayList<String> sortByParam = new ArrayList<>();
            ArrayList<String> orderByParam = new ArrayList<>();
            sortByParam.add(sortBy.getValue());
            orderByParam.add(orderBy.toString());

            params.put("sortby", sortByParam);
            params.put("sortorder", orderByParam);
        }

        return params;
    }

    /**
     * Gets key that represents the caching of the returned objects. This is useful in the event
     * that multiple of the same APIs are made with different query parameters.
     *
     * @return The unique caching key for the data being stored.
     */
    public String getCachingKey() {
        return cachingKey;
    }

    /**
     * Gets the current pagination limit in these params.
     * @return The current limit.
     */
    public long getLimit() {
        return limit;
    }

    /**
     * Gets the current pagination offset in these params.
     * @return The current offset.
     */
    public long getOffset() {
        return offset;
    }

    /**
     * Gets the current labels to filter by in these params.
     * @return The labels currently being filtered by.
     */
    public ArrayList<String> getLabels() {
        return labels;
    }

    /**
     * Gets the current field being sorted by in these params.
     * @return The current sort field.
     */
    public SortByEnumeratable getSortBy() {
        return sortBy;
    }

    /**
     * Gets the current sort order in these params.
     * @return The current sort order.
     */
    public OrderBy getOrderBy() {
        return orderBy;
    }

    /**
     * Sets paging options, based on a starting offset to some limit.
     *
     * @param limit The maximum number of returned objects.
     * @param offset The offset of where the next X number of response objects will be returned
     *               from where X is the limit.
     */
    public void setPaging(long limit, long offset){
        this.limit  = limit;
        this.offset = offset;
    }
}
