package com.flybits.commons.library.caching;

import android.arch.lifecycle.LiveData;
import android.arch.lifecycle.Observer;
import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.flybits.internal.db.CommonsDatabase;
import com.flybits.internal.db.models.CachingEntry;

import java.util.ArrayList;
import java.util.List;

/**
 * The {@code FlybitsUIListObserver} class is responsible for creating an {@code Observer} that
 * listens to local Database changes. Once the {@code Observer} detects that there is a change in
 * data in one of the Database fields will be will notify the application using the
 * {@link DataChanged#add(String, DataChanged)} callback. Within this callback method the
 * application will be able to perform all the necessary UI changes.
 *
 * <p>The {@code FlybitsUIListObserver} is responsible for queries that are associated to a List of
 * object rather than a single object. If you are interested in Single object manipulation then you
 * should look at the {@link FlybitsUIObjectObserver} class.</p>
 *
 * @param <T> The object that {@code Observer} is observing for content changes.
 */
public abstract class FlybitsUIListObserver<T> {

    private DataChanged<T> callback;
    private LiveData<List<T>> item;
    private Observer<List<T>> observer;

    private List<T> itemsLoaded;
    private Context context;

    /**
     * Default Constructor used to initialize the {@code FlybitsUIListObserver} object.
     *
     * @param context The Context of the application
     */
    public FlybitsUIListObserver(@NonNull Context context){
        this.context = context;
        itemsLoaded = new ArrayList<>();
    }

    /**
     * Adds a {@link DataChanged} callback that is responsible for indicating to the UI application
     * when data has changes. When data does change then the UI should make the appropriate changes.
     *
     * @param key The caching used to retrieve entities for.
     * @param callback The {@link DataChanged} callback that is initiated when there is a change in
     *                 data through the Database.
     */
    public void add(@NonNull final String key, DataChanged<T> callback){
        this.callback   = callback;

        new Thread(new Runnable() {
            public void run() {

                List<CachingEntry> listOfCacheObjects = CommonsDatabase.getDatabase(context).cachingEntryDAO().getByKey(key);
                List<String> ids                      = new ArrayList<>();
                for(CachingEntry entry : listOfCacheObjects){
                    ids.add(entry.getContentId());
                }

                onLoad(ids);
            }
        }).start();
    }

    /**
     * Get the context of the application.
     *
     * @return The Context of the application.
     */
    public Context getContext() {
        return context;
    }

    /**
     * Get the current {@code LiveData} object.
     *
     * @return The current {@code LiveData} object, if {@code setItems(LiveData)}
     * has not been called, then this method will return null.
     */
    @Nullable public LiveData<List<T>> getData(){
        return item;
    }

    /**
     * This method is called once the Cache is ready to be loaded. Once it is ready to be loaded the
     * implementing class must then load the correct entity based on the provided {@code cachingIds}
     *
     * @param cachingIds The list of Ids that should be loaded
     */
    public abstract void onLoad(List<String> cachingIds);

    /**
     * Removes the {@code Observer} from reading changes to the List of objects within the database.
     * In most cases, this should occur when the UI that is listening for changes is destroyed.
     */
    public void remove(){
        if (item != null && item.hasActiveObservers()){
            item.removeObserver(observer);
        }
    }

    /**
     * Return a list of already loaded items. If no items have been loaded or there are no item
     * cached then this will return a empty list of items.
     *
     * @return A list of items that are cached and loaded.
     */
    public List<T> getItems(){
        return itemsLoaded;
    }

    /**
     * Sets the {@code LiveData} list that is being observed for changes.
     *
     * @param item The {@code LiveData} list that is be observed for changes.
     */
    public void setItems(@NonNull LiveData<List<T>> item){
        this.item   = item;
        this.observer   = new Observer<List<T>>() {
            @Override
            public void onChanged(@Nullable List<T> items) {
                if (callback != null) {
                    callback.onLoad(items);
                }
            }
        };
        this.item.observeForever(observer);
    }
    
    /**
     * The {@code DataChanged} is a callback responsible for indicating to the UI application when
     * data within the database, for which this class is observing, has changed.
     *
     * @param <T> The object that {@code Observer} is observing for content changes.
     */
    public interface DataChanged<T> {

        /**
         * Method responsible for indicating to the UI application that a change in the observed
         * data has changed.
         *
         * @param data The list of objects from the database. Some of these objects will have
         *             changed while other have not.
         */
        void onLoad(List<T> data);
    }
}
