package com.flybits.commons.library.caching;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;

import com.flybits.internal.db.CommonsDatabase;
import com.flybits.internal.db.models.CachingEntry;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * The {@code FlybitsCacheLoader} is responsible for obtaining cache results for a specific entity.
 * This is an important aspect as it will allow application to display content to the user that has
 * been previously saved rather than a loading screen.
 *
 * @param <T> The entity that cached content is being retrieved for.
 */
public abstract class FlybitsCacheLoader<T>{

    private CacheLoader callback;
    private ExecutorService executorService;
    private Context context;

    /**
     * Default constructor
     *
     * @param context The Context of the application.
     */
    public FlybitsCacheLoader(@NonNull Context context){
        this.context = context;
    }

    /**
     * Get the {@code Context} of the application.
     *
     * @return The Context of the application.
     */
    public Context getContext(){
        return context;
    }

    /**
     * Abstract method that is responsible for retrieving cache data from the local DB. This method
     * will be called in a separate thread so no threading needs to take place.
     *
     * @param cachedIds The list of cached ids that should be retrieved.
     * @return The cached list of items associated to the entity defined in the class generic.
     */
    public abstract List<T> load(List<String> cachedIds);

    /**
     * Retrieves the cached data defined within the {@link #load(List)} method and loads the based
     * through the {@code callback} parameter's callback methods.
     *
     * @param cachingKey The caching key which should be used to retrieve a specific entity list.
     * @param callback The {@link CacheLoader} callback that indicates when cached data is loaded
     *                 and then that loading was cancelled.
     */
    public void get(@NonNull final String cachingKey, final CacheLoader<T> callback){

        this.callback   = callback;
        final Handler handler = new Handler(Looper.getMainLooper());
        executorService = Executors.newSingleThreadExecutor();
        executorService.execute(new Runnable() {
            @Override
            public void run() {


                List<CachingEntry> listOfCacheObjects = CommonsDatabase.getDatabase(context).cachingEntryDAO().getByKey(cachingKey);
                List<String> ids                      = new ArrayList<>();
                for(CachingEntry entry : listOfCacheObjects){
                    ids.add(entry.getContentId());
                }

                final List<T> loadedList  = load(ids);
                if (callback != null) {
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            if (loadedList != null) {
                                callback.onLoad(loadedList);
                            } else {
                                callback.onLoad(new ArrayList<T>());
                            }
                        }
                    });
                }
            }
        });
    }

    /**
     * Cancels the cache loading in the event that it is taking too long. This is a manual call the
     * application.
     */
    public void cancel(){
        if (executorService != null && !executorService.isShutdown()){
            executorService.shutdownNow();

            if (callback != null){
                callback.onCancel();
            }
        }
    }

    /**
     * Callback used to indicate when cached information is loaded from the DB or when the request
     * is cancelled by the application.
     *
     * @param <T> The entity that cached content is being retrieved for.
     */
    public interface CacheLoader<T>{

        /**`
         * Indicates that the caching request has been cancelled by the application. This should
         * happen if the application feels like the cache request is taking too long or if the
         * network request finished before the cache loading which is highly unlikely.
         */
        void onCancel();

        /**
         * Indicates the cache retrieval has completed and the list has been retrieved.
         *
         * @param listToLoad The list of loaded data.
         */
        void onLoad(List<T> listToLoad);
    }
}