package com.flybits.commons.library.api.results;


import android.content.Context;
import android.support.annotation.NonNull;

import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.models.internal.PagedResponse;
import com.flybits.commons.library.models.internal.Pagination;
import com.flybits.commons.library.models.internal.QueryParameters;
import com.flybits.commons.library.models.internal.Result;

import java.util.ArrayList;
import java.util.concurrent.ExecutorService;

/**
 * The {@code PagedResult} class is used to query the server when the response is excepted to have
 * a pagination object. Through this class, the application will be able to call the
 * {@link #getMore(PagedResultCallback)} method order to retrieve the next set of results.
 *
 * @param <T> The object that is expected to be returned as a list along with the pagination
 *           information.
 */
public abstract class PagedResult<T> extends FlybitsResult {

    private long currentOffset;
    private long totalItems;
    private long limit;
    private PagedResultCallback<T> callback;
    private QueryParameters parameters;

    /**
     * Constructor used to initialize the {@code PagedResult} object.
     *
     * @param context The context of the application.
     * @param callback The {@link BasicResultCallback} that indicates whether or not the network
     *                 request was successful or not.
     */
    public PagedResult(@NonNull Context context, PagedResultCallback<T> callback){
        super(context);
        limit           = 0;
        currentOffset   = 0;
        totalItems      = 0;
        this.callback   = callback;
    }

    /**
     * Constructor used to initialize the {@code BasicResult} object.
     *
     * @param context The context of the application.
     * @param callback The {@link BasicResultCallback} that indicates whether or not the network
     *                 request was successful or not.
     * @param service The {@link ExecutorService} that is used to run the network request thread.
     */
    public PagedResult(@NonNull Context context, PagedResultCallback<T> callback, @NonNull ExecutorService service){
        super(context, service);
        limit           = 0;
        currentOffset   = 0;
        totalItems      = 0;
        this.callback   = callback;
    }

    /**
     * Makes a network request to get the next set of items if there are more items to be retrieved.
     *
     * @param callback The {@link PagedResultCallback} that indicates whether or not more items where
     *                 able to be retrieved.
     */
    public PagedResult getMore(PagedResultCallback<T> callback){

        if (hasMore()) {
            parameters.setPaging(limit, currentOffset);
            return getMore(getContext(), parameters, callback);
        }else{
            if (callback != null) {
                callback.onLoadedAllItems();
            }
            return this;
        }
    }

    /**
     * Get the extra parameters used for this request.
     *
     * @return The {@link QueryParameters} associated to this network request.
     */
    public QueryParameters getParameters(){
        return parameters;
    }

    /**
     * Indicates whether or not more items can be retrieved from the server.
     *
     * @return true if more items can be retrieved, false if all items have already been retrieved.
     */
    boolean hasMore(){
        return limit < totalItems && currentOffset < totalItems;
    }

    /**
     * Sets the {@link FlybitsException} that was thrown when request more items to be loaded.
     *
     * @param e The {@link FlybitsException} that caused the network to fail.
     * @return true if the error has been set, false otherwise
     */
    public boolean setFailed(FlybitsException e) {

        if (super.setFailed(e)) {
            if (callback != null) {
                callback.onException(e);
            }
            return true;
        }
        return false;
    }

    /**
     * Sets the result of the network request used to retrieve more items. In all cases, this should
     * include the items retrieved along with the returned {@link Pagination} object.
     *
     * @param parameters The {@link QueryParameters}, if any, that that indicate any additional
     *                   filters used for this network request.
     * @param items The items returned from the {@link #getMore(PagedResultCallback)} query.
     * @param pagination The {@link Pagination} object that can be used find out how many more items
     *                   are available to be retrieved.
     */
    protected void setSuccess(QueryParameters parameters, @NonNull ArrayList<T> items, @NonNull Pagination pagination){
        this.parameters = parameters;
        this.limit      = pagination.getLimit();
        this.totalItems = pagination.getTotalRecords();
        currentOffset   = (items != null) ? pagination.getOffset() + items.size() : 0;

        if (setSuccess() && items != null && callback != null) {
            callback.onSuccess(items);
            if (!hasMore()) {
                callback.onLoadedAllItems();
            }
        }
    }

    /**
     * Set the {@link Result} of the network request that contains the object returned from
     * the network request.
     *
     * @param result The {@link Result} that contains a {@link PagedResponse} object which will
     *               contain the list of items as well as the {@link Pagination} object that
     *               indicates paging information about this request.
     * @param params The {@link QueryParameters} that contains the filter information for this
     *               {@link Result}.
     */
    public void setResult(@NonNull Result<PagedResponse<T>> result, QueryParameters params){
        if (result.getStatus() == RequestStatus.COMPLETED && result.getResult() != null){
            setSuccess(params, result.getResult().getItems(), result.getResult().getPagination());
        }else{
            setFailed(result.getException());
        }
    }

    /**
     * Responsible for retrieving more item of the desired query. Each query should implement it's
     * own {@code #getMore(Context, QueryParameters, QueryCallback)} method as each query may have it's
     * own logic for retrieving items.
     *
     * @param context The context of the application.
     * @param parameters The {@link QueryParameters}, if any, that that indicate any additional
     *                   filters used for this network request.
     * @param callback The {@link PagedResultCallback} that indicates whether or not more items where
     *                 able to be retrieved.
     */
    protected abstract <K extends QueryParameters> PagedResult getMore(Context context, K parameters,
                                    PagedResultCallback<T> callback);
}
