package com.flybits.commons.library.api;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;

import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.analytics.AnalyticsOptions;
import com.flybits.commons.library.analytics.Properties;
import com.flybits.commons.library.api.idps.FlybitsIDP;
import com.flybits.commons.library.api.idps.IDP;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.ConnectionResult;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ConnectionResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.deserializations.DeserializeLogin;
import com.flybits.commons.library.deserializations.DeserializePagedResponse;
import com.flybits.commons.library.deserializations.DeserializeProject;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.exceptions.InvalidFlybitsManagerException;
import com.flybits.commons.library.http.HttpDefaultClass;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.models.Jwt;
import com.flybits.commons.library.models.Project;
import com.flybits.commons.library.models.ProjectParameters;
import com.flybits.commons.library.models.User;
import com.flybits.commons.library.models.internal.PagedResponse;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.commons.library.models.results.ProjectsResult;
import com.flybits.internal.db.CommonsDatabase;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import static com.flybits.commons.library.SharedElements.getSavedJWTToken;

/**
 * The {@code FlybitsManager} class is responsible for defining all {@link FlybitsScope}s as well as
 * performing all {@link User} operations such as logging in, logging out, disabling user's account,
 * user forgotten password, and refreshing the logged in user's JWT.
 */
public class FlybitsManager {

    public static final String AUTHENTICATION_API      = "/sso/auth";
    static final String PROJECTS_API            = "/kernel/projects";
    static final String PROJECT_ENDPOINT        = AUTHENTICATION_API+"/project";
    static final String DISCONNECT_ENDPOINT     = AUTHENTICATION_API+"/logout";

    public static boolean IS_DEBUG = false;

    private AnalyticsOptions analyticsOptions;
    private Context context;
    private boolean enableAnalytics;
    private IDP idProvider;
    private boolean isDebug;
    private ArrayList<String> languageCodes;
    private Set<FlybitsScope> listOfScopes;
    private String projectId;

    /**
     * Constructor used to define the {@code FlybitsManager} based on the {@link Builder}
     * attributes.
     *
     * @param builder The {@link Builder} that defines all the attributes about the
     * {@code FlybitsManager}.
     */
    private FlybitsManager(Builder builder){
        listOfScopes                = builder.listOfScopes;
        context                     = builder.mContext;
        isDebug                     = builder.isDebug;
        enableAnalytics             = builder.enableAnalytics;
        languageCodes               = builder.languageCodes;
        idProvider                  = builder.idProvider;
        projectId                   = builder.projectId;

        if (languageCodes.size() == 0){
            languageCodes.add("en");
        }

        IS_DEBUG    = isDebug;
        if (enableAnalytics) {
            if (builder.analyticsOptions != null) {
                analyticsOptions = builder.analyticsOptions;
            } else {
                analyticsOptions = new AnalyticsOptions.Builder(context)
                        .setStorageType(AnalyticsOptions.StorageType.SQLITE_DB)
                        .setUploadServiceTime(60, 60, TimeUnit.HOURS)
                        .build();
            }
        }

        if (projectId != null){
            SharedElements.setProjectID(context, projectId);
        }
    }

    /**
     * The {@code Builder} class is responsible for setting all the attributes associated to the
     * {@link FlybitsManager}.
     */
    public static final class Builder {

        private final Set<FlybitsScope> listOfScopes;
        private Context mContext;
        private IDP idProvider;
        private boolean isDebug;
        private boolean enableAnalytics;
        private AnalyticsOptions analyticsOptions;
        private ArrayList<String> languageCodes;
        private String projectId;

        /**
         * Constructor that initializes all the objects within the {@code Builder} class.
         *
         * @param context The {@link Context} of the application.
         */
        public Builder(@NonNull Context context) {
            this.mContext       = context;
            this.listOfScopes   = new HashSet<>();
            this.languageCodes  = new ArrayList<>();
            try {
                this.languageCodes.add(Locale.getDefault().getLanguage());
            }catch (Exception e){}
        }

        /**
         * Add a {@link FlybitsScope} to the {@link FlybitsManager}. The {@link FlybitsScope}
         * defines the SDK and all the properties associated to it.
         *
         * @param scope The {@link FlybitsScope} that is associated to this application.
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder addScope(@NonNull FlybitsScope scope) {
            listOfScopes.add(scope);
            return this;
        }

        /**
         * Build the {@link FlybitsManager} object that is used for defining the SDKs that are
         * associated to the application.
         *
         * @return {@link FlybitsManager} object which can be referenced by the SDK to get specific
         * information about the SDKs defined within the application.
         * @throws InvalidFlybitsManagerException if the {@link Builder#addScope(FlybitsScope)}
         * method was not called.
         */
        public FlybitsManager build() throws InvalidFlybitsManagerException {
            checkIfFieldsSet();
            return new FlybitsManager(this);
        }

        /**
         * Enables analytics to be automatically recorded for the SDK, as well as initialize the
         * {@link Analytics} component in the event that the application would like to use it as
         * Flybits Analytics as well.
         *
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder enableAnalytics(){
            this.enableAnalytics    = true;
            return this;
        }

        /**
         * Sets a custom Analytics configuration over the default one. Create one using {@link AnalyticsOptions.Builder}.
         *
         * @param options The {@link AnalyticsOptions} that will override the default options.
         *
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder overrideAnalyticsOptions(AnalyticsOptions options) {
            this.analyticsOptions   = options;
            return enableAnalytics();
        }

        /**
         * Sets the {@link IDP} which should be used sign into the Flybits Account Manager.
         *
         * @param idp The {@link IDP} that should be used for registering contextual information to
         *            the account.
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setAccount(IDP idp){
            this.idProvider = idp;
            return this;
        }

        /**
         * Sets the Flybits Android commons SDK in Debug mode. While in debug mode all logs will be
         * displayed within the logcat of your Android Studio including any errors/exceptions.
         *
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setDebug(){
            this.isDebug    = true;
            return this;
        }

        /**
         * Sets the Language that the SDK should parse the localized values with.
         * @param languageCode The 2-letter code, such as "en", that indicates which language should
         *                     be used for parsing localized values.
         *
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setLanguage(@NonNull String languageCode){
            languageCodes.clear();
            languageCodes.add(languageCode);
            return this;
        }

        /**
         * Sets the project identifier of the application.
         *
         * @param projectId The unique GUID that represents the Project Id for the application.
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setProjectId(@NonNull String projectId){
            this.projectId = projectId;
            return this;
        }

        private void checkIfFieldsSet()throws InvalidFlybitsManagerException {

            Properties properties = new Properties();
            if (listOfScopes.size() == 0){
                if (Analytics.isInitialized()) {
                    Analytics.logEventFlybits(AnalyticsCommonsEvents.EVENT_MISSING_SCOPE, properties);
                }
                throw new InvalidFlybitsManagerException("You must have at least 1 scope set to create a valid FlybitsManager object");
            }

            if (languageCodes.size() != 1 && languageCodes.get(0).length() != 2){
                if (Analytics.isInitialized()) {
                    Analytics.logEventFlybits(AnalyticsCommonsEvents.EVENT_INVALID_LANGUAGE_CODE, properties);
                }
                throw new InvalidFlybitsManagerException("Your language must be a 2-letter code. Make sure you call setLanguage(String)");
            }
        }
    }

    /**
     * Indicate that the logcat should display logs from the SDK.
     */
    public static void setDebug(){
        IS_DEBUG    = true;
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method.
     *
     * @param callback The response callback to run after a connection is made or fails.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public ConnectionResult connect(final ConnectionResultCallback callback) {
        return connect(callback, true);
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method.
     *
     * @param idp The {@link IDP} that should be used for registering contextual information to
     *            the account.
     * @param callback The response callback to run after a connection is made or fails.
     * @param autoUseManifestProject If true, the project id defined in the manifest will be loaded
     *                               automatically.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public ConnectionResult connect(final IDP idp, final ConnectionResultCallback callback, final boolean autoUseManifestProject) {
        idProvider  = idp;
        return connect(callback, autoUseManifestProject);
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method.
     *
     * @param idp The {@link IDP} that should be used for registering contextual information to the
     *            account.
     * @param callback The response callback to run after a connection is made or fails.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public ConnectionResult connect(final IDP idp, final ConnectionResultCallback callback) {
        idProvider  = idp;
        return connect(callback, true);
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method. If autoUseManifestProject is true,
     * the project to load will be pulled from the manifest. Otherwise it will be in a no-project
     * state. This can only be done with the Flybits IDP.
     *
     * @param callback The response callback to run after a connection is made or fails.
     * @param autoUseManifestProject If true, the project id defined in the manifest will be loaded
     *                               automatically.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public ConnectionResult connect(final ConnectionResultCallback callback, final boolean autoUseManifestProject){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ConnectionResult query = new ConnectionResult(context, callback, executorService);

        executorService.execute(new Runnable() {
            @Override
            public void run() {

                try {
                    //TODO: first check if JWT is empty
                    final Result jwtResult    = FlyJWT.refreshJWT(context);
                    if (jwtResult.getStatus()   == RequestStatus.COMPLETED){
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                query.setResult(jwtResult);
                            }
                        });
                    }else{

                        if (idProvider != null) {

                            String body     = idProvider.getRequestBody(context, autoUseManifestProject).toString();
                            String url      = AUTHENTICATION_API + idProvider.getAuthenticationEndPoint();
                            final Result<User> authenticatedUser = FlyAway.post(context, url, body, new DeserializeLogin(), "FlybitsManager.connect", User.class);

                            if (authenticatedUser.getStatus() == RequestStatus.COMPLETED) {

//                            MQTTInstance.getInstance(context).connect(context);
                                CommonsDatabase.getDatabase(context).userDao().insert(authenticatedUser.getResult());
                                SharedElements.setConnectedIDP(context, idProvider.getProvider());

                                if (!Analytics.isInitialized() && enableAnalytics) {
                                    Analytics.initialize(analyticsOptions);
                                    Properties properties = new Properties();
                                    properties.addProperty(AnalyticsCommonsEvents.PARAM_IDP_TYPE, idProvider.getClass().getCanonicalName());
                                    Analytics.logEventFlybits(AnalyticsCommonsEvents.EVENT_SDK_CONNECTED, properties);
                                }

                                for (FlybitsScope scope : listOfScopes) {
                                    scope.onConnected(context, authenticatedUser.getResult());
                                }
                            }
                            handler.post(new Runnable() {
                                @Override
                                public void run() {
                                    query.setResult(authenticatedUser);
                                }
                            });

                            SharedElements.setLocalization(context, languageCodes);
                        }else{
                            handler.post(new Runnable() {
                                @Override
                                public void run() {
                                    query.setNotConnected();
                                }
                            });
                        }
                    }
                } catch (final FlybitsException e) {
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            query.setFailed(e);
                        }
                    });
                }
            }
        });

        return query;
    }

    /**
     * Called to get all {@code Project}s that exist on this account.
     * @param params Defines paging info.
     * @param callback The callback used to collect the Project objects, or an error if one occured.
     * @return The {@link ProjectsResult} object can be used to continue paging if more items can
     * be fetched.
     */
    public ProjectsResult getProjects(final ProjectParameters params, final PagedResultCallback<Project> callback)
    {
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ProjectsResult result = new ProjectsResult(context, callback, executorService);

        executorService.execute(new Runnable() {
            @Override
            public void run() {

                try{
                    boolean isConnnected = !getSavedJWTToken(context).equals("");
                    if (isConnnected && getIDP() instanceof FlybitsIDP){
                        DeserializeProject singleDeserializaer = new DeserializeProject();
                        DeserializePagedResponse<Project> deserializer = new DeserializePagedResponse<Project>(singleDeserializaer);

                        final Result<PagedResponse<Project>> projectList = FlyAway.get(context, PROJECTS_API, params, deserializer, "FlybitsManager.getProjects");

                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                result.setResult(projectList, params);
                            }
                        });
                    }
                    else
                    {
                        throw new FlybitsException("Either not yet connected to Flybits, or you are not using the Flybits IDP. GetProjects() only works with the Flybits IDP.");
                    }
                }catch (final FlybitsException e){

                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            result.setFailed(e);
                        }
                    });
                }
            }
        });

        return result;
    }

    /**
     * Sets the project id to another project, refreshing the current session.
     * @param projectId The project id to bind to.
     * @param callback An interface to resolve successes and failures of this call.
     * @return The {@link BasicResult} is an object that is returned from the Server after a HTTP
     * request is made.
     */
    public BasicResult bindProject(final String projectId, final BasicResultCallback callback)
    {
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(context, callback, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {

                try {

                    String currentJwtToken = getSavedJWTToken(context);
                    Jwt currentJwt = Jwt.decodeJWTToken(currentJwtToken);

                    if (currentJwt.isProjectBound())
                        throw new FlybitsException("There is already a project bound, the client must logout first.");

                    HashMap<String, String> headers = new HashMap<>();
                    String url = FlybitsAPIConstants.constructGatewayURL(context, String.format("%s?projectId=%s", PROJECT_ENDPOINT, projectId));

                    try {
                        final Result bound = new HttpDefaultClass.Builder(context, true, url)
                                .addHeaders(headers)
                                .get()
                                .getResponse();

                        if (bound != null && bound.getStatus() == RequestStatus.COMPLETED) {
                                if (!Analytics.isInitialized() && enableAnalytics) {
                                    Analytics.initialize(analyticsOptions);
                                }
                        }

                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                request.setResult(bound);
                            }
                        });

                    } catch (Exception e) {
                            throw new FlybitsException("Error Connecting to Flybits Server: FlybitsManager.bindProject");
                    }

                } catch (final FlybitsException e) {

                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setFailed(e);
                        }
                    });
                }
            }
        });
        return request;
    }

    /**
     * Clears all information associated this instance of the SDK. If you use this method
     * persistence between app launches will not be maintained, therefore it is highly recommended
     * that you use the {@link #disconnect(BasicResultCallback)} method over this {@code destroy()}
     * one.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult destroy(final BasicResultCallback callback){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(context, callback, executorService);

        executorService.execute(new Runnable() {
            @Override
            public void run() {

                try {
                    if (Analytics.isInitialized()) {
                        Analytics.destroy(context);
                    }

                    String jwtToken = SharedElements.getSavedJWTToken(context);
                    final Result deleteUser = FlyAway.delete(context, User.ME_ENDPOINT, "FlybitsManager.destroy", null);
                    if (deleteUser.getStatus() == RequestStatus.COMPLETED) {
                        clearUserInformation(context);
                        CommonsDatabase.getDatabase(context).userDao().delete();
                        for (FlybitsScope scope : listOfScopes) {
                            scope.onAccountDestroyed(context, jwtToken);
                        }
                    }

                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setResult(deleteUser);
                        }
                    });
                } catch (final FlybitsException e) {
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setFailed(e);
                        }
                    });
                }
            }
        });
        return request;
    }

    /**
     * The {@code disconnect} method is responsible for clearing the server sessions with Flybits.
     * In addition, once a logout is successful it will notify all the registered
     * {@link FlybitsScope}s through the {@link FlybitsScope#onDisconnected(Context, String)}
     * method.
     *
     * @param callback The callback used to indicate whether or not the disconnection was
     * successful or not.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult disconnect(final BasicResultCallback callback){
        return disconnect(callback, false);
    }

    /**
     * The {@code disconnect} method is responsible for clearing the server sessions with Flybits.
     * In addition, once a logout is successful it will notify all the registered
     * {@link FlybitsScope}s through the {@link FlybitsScope#onDisconnected(Context, String)} method.
     *
     * @param callback The callback used to indicate whether or not the disconnection was
     * successful or not.
     * @param disconnectOnException true if the disconnect should clear all data even if the network
     *                              request was unsuccessful, false otherwise.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult disconnect(final BasicResultCallback callback, final boolean disconnectOnException){
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(context, callback, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {

                try {
                    if (!Analytics.isInitialized()) {
                        Analytics.destroy(context);
                    }

                    String jwtToken = SharedElements.getSavedJWTToken(context);
                    final Result<String> disconnected = FlyAway.post(context, DISCONNECT_ENDPOINT, "", null, "FlybitsManager.disconnect", null);

                    if (disconnected.getStatus() == RequestStatus.COMPLETED || disconnectOnException) {
                        clearUserInformation(context);
                        CommonsDatabase.getDatabase(context).userDao().delete();
                    }

                    if (disconnected.getStatus() == RequestStatus.COMPLETED) {
                        if (Analytics.isInitialized()) {
                            Analytics.logEventFlybits(AnalyticsCommonsEvents.EVENT_SDK_DISCONNECTED, null);
                        }
                        for (FlybitsScope scope : listOfScopes) {
                            scope.onDisconnected(context, jwtToken);
                        }
                    }

                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setResult(disconnected);
                        }
                    });
                } catch (final FlybitsException e) {
                    if (disconnectOnException) {
                        clearUserInformation(context);
                        CommonsDatabase.getDatabase(context).userDao().delete();
                    }

                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setFailed(e);
                        }
                    });
                }
            }
        });
        return request;
    }

    /**
     * The {@code getUser} method is responsible for retrieving user information about the Flybits
     * {@link User}. The {@link User} contains basic personal attributes such as
     * {@link User#firstName} and {@link User#lastName} as well as Flybits information such as
     * {@link User#id} and {@link User#deviceID}.
     *
     * @return The {@link ObjectResult} is an object that is returned from the Server after a HTTP
     * request is made.
     * @deprecated Please use the more optimized {@link User#getSelf} static method.
     */
    public static ObjectResult<User> getUser(final Context context){
        return User.getSelf(context, null);
    }

    /**
     * The {@code getUser} method is responsible for retrieving user information about the Flybits
     * {@link User}. The {@link User} contains basic personal attributes such as
     * {@link User#firstName} and {@link User#lastName} as well as Flybits information such as
     * {@link User#id} and {@link User#deviceID}.
     *
     * @return The {@link ObjectResult} is an object that is returned from the Server after a HTTP
     * request is made.
     * @deprecated Please use the more optimized {@link User#getSelf} static method.
     */
    public static ObjectResult<User> getUser(final Context context, ObjectResultCallback<User> callback){
        return User.getSelf(context, callback);
    }

    /**
     * Check to see if the SDK currently has established a connection with the Flybits Server.
     *
     * @param context The Context of the application.
     * @param confirmThroughNetwork Indicates whether or not a confirm should take place with the
     *                              server. If false, this will simply check is a token is saved
     *                              locally, this token may however not be valid. If true, it will
     *                              confirm with the server that it is valid.
     * @param callback The callback that indicates whether or not the SDK is currently connected to
     *                 Flybits.
     * @return The {@link ConnectionResult} which is executing the network request (if any) to
     * determine whether or not the connection is valid.
     */
    public static ConnectionResult isConnected(@NonNull final Context context,
                                               final boolean confirmThroughNetwork,
                                               @NonNull final ConnectionResultCallback callback){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        ConnectionResult query   = new ConnectionResult(context, callback, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {

                if (confirmThroughNetwork && !getSavedJWTToken(context).equals("")) {

                    try {
                        final Result getJWT = FlyJWT.refreshJWT(context);

                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                if (getJWT.getStatus() == RequestStatus.NOT_CONNECTED) {
                                    callback.notConnected();
                                } else if (getJWT.getStatus() == RequestStatus.COMPLETED) {
                                    callback.onConnected();
                                } else {
                                    callback.onException(getJWT.getException());
                                }
                            }
                        });
                    } catch (final FlybitsException e) {
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                callback.onException(e);
                            }
                        });
                    }
                } else if (getSavedJWTToken(context).equals("")) {
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.notConnected();
                        }
                    });

                } else {
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.onConnected();
                        }
                    });
                }
            }
        });
        return query;
    }

    static void clearUserInformation(Context context) {

//        MQTTInstance.getInstance(context).disconnect(context);
        SharedPreferences preferences = SharedElements.getPreferences(context);
        SharedPreferences.Editor editor = preferences.edit();
        editor.clear();
        editor.apply();
    }

    IDP getIDP(){
        return idProvider;
    }

    Set<FlybitsScope> getScopes(){
        return listOfScopes;
    }

    boolean isDebug(){return isDebug;}

    boolean isAnalyticsEnabled(){return enableAnalytics;}

}
