package com.flybits.commons.library.utils;

import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Bundle;
import android.provider.Settings;
import android.support.annotation.NonNull;

import com.flybits.commons.library.BuildConfig;
import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.logging.Logger;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Arrays;

public class Utilities {

    /**
     * Converts a string representation of language codes into a list of language codes.
     *
     * @param languageCodeAsString The string representation of a list of language codes.
     * @return the localization codes as a list, if there are no localization codes set then an
     * empty list will be returned.
     */
    public static ArrayList<String> convertLocalizationStringToList(String languageCodeAsString){

        if (languageCodeAsString == null || languageCodeAsString.length() == 0){
            return new ArrayList<>();
        }

        String[] listOfCodes = languageCodeAsString.split(",");
        return new ArrayList<>(Arrays.asList(listOfCodes));
    }

    /**
     * Converts a list of language codes into a single string representation that can be attached
     * to the header of any request.
     *
     * @param listOfCodes The list of codes to convert to String representations.
     * @return the localization codes as String, if there are no localization codes set then an
     * empty string is returned.
     */
    public static String convertLocalizationCodeToString(ArrayList<String> listOfCodes){
        if (listOfCodes != null && listOfCodes.size() > 0){

            String languageCodeString = listOfCodes.get(0);
            for (int i = 1; i < listOfCodes.size(); i++){
                languageCodeString = languageCodeString.concat("," + listOfCodes.get(i));
            }
            return languageCodeString;
        }
        return "";
    }

    /**
     * Get the unique device identifier for the device using this application.
     *
     * @param context The current state of the application.
     * @return String representation for the unique device identifier.
     */
    public static String getDeviceID(@NonNull Context context) {
        return Settings.Secure.getString(context.getContentResolver(), Settings.Secure.ANDROID_ID);
    }

    /**
     * Get the version of the Flybits Commons SDK
     *
     * @return The string representation of the Flybits Commons SDK Version
     */
    public static String getSDKVersion() {
        return BuildConfig.FlybitsVersion;
    }

    /**
     * Gets the data associated to the device such as the make, model, and OS Version number. This
     * information is sent with every HTTPS request to the user for analytics reasons.
     *
     * @return JSON representation of the User Agent model for all HTTPS requests.
     */
    public static String getUserAgentAsJSON(Context context) {

        JSONObject obj = new JSONObject();
        try {
            obj.put("make", Build.MANUFACTURER);
        }catch (JSONException e){}

        try {
            obj.put("model", Build.MODEL);
        }catch (JSONException e){}

        try {
            obj.put("osVersion", String.valueOf(Build.VERSION.SDK_INT));
        }catch (JSONException e){}

        try {
            obj.put("deviceType", "Android");
        }catch (JSONException e){}

        try {
            obj.put("physicalDeviceId", getDeviceID(context));
        }catch (JSONException e){}

        try {
            obj.put("sdkVersion", getSDKVersion());
        }catch (JSONException e){}


        try {
            android.content.pm.ApplicationInfo ai = context.getPackageManager()
                    .getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);

            Bundle bundle = ai.metaData;
            if (bundle.containsKey("com.flybits.device.name")) {
                String name    = bundle.getString("com.flybits.device.name");
                obj.put("name", name);
            }else{
                obj.put("name", Build.MANUFACTURER+"-"+Build.MODEL);
            }
        }catch (Exception e) {
            Logger.exception("getUserAgentAsJSON", e);
        }

        return obj.toString();
    }
}
