package com.flybits.commons.library.analytics;

import android.content.Context;
import android.support.annotation.NonNull;

import java.util.ArrayList;
import java.util.concurrent.TimeUnit;

/**
 * The {@code AnalyticsOptions} class is used to store all options that may have been configured by
 * the {@link AnalyticsOptions.Builder}.
 */
public class AnalyticsOptions {

    public enum StorageType{

        /**
         * Events will be stored in an Sqlite Database. If one cannot be made for some reason,
         * it will fallback to memory storage.
         */
        SQLITE_DB(0, "sqlite_db"),

        /**
         * Events will be stored in the app's physical storage space. If the file cannot be created
         * it will fallback to memory storage.
         */
        BINARY(1, "binary"),

        /**
         * Events are stored in memory only. Warning: unflushed events will be lost if the app is
         * killed.
         */
        MEMORY_ONLY(2, "memory");

        /**
         * Events are stored in a custom implementation of QueueStorage
         */

        private final int key;
        private final String value;

        /**
         * Constructor that defines the key for each {@code StorageType} option.
         *
         * @param key the String value representing each {@code StorageType} option.
         */
        StorageType(int key, String value) {
            this.key    = key;
            this.value  = value;
        }

        /**
         * Get the String representation for the {@code StorageType} option.
         *
         * @return String representation of the {@code StorageType} option.
         */
        public int getKey() {
            return this.key;
        }

        /**
         * Get the String representation for the {@code StorageType} option.
         *
         * @return String representation of the {@code StorageType} option.
         */
        public String getValue() {
            return this.value;
        }

        /**
         * Get the {@code StorageType} enum value corresponding to an String representation.
         *
         * @param value the String representation of the {@code StorageType} enum.
         *
         * @return The {@code StorageType} enum for the String representation.
         */
        public static StorageType fromValue(String value) {
            for(StorageType type : StorageType.values()) {
                if(type.getValue().equalsIgnoreCase(value)) {
                    return type;
                }
            }
            return MEMORY_ONLY;
        }

    }

    private Context context;
    private String customDeviceId;
    private StorageType storageType;
    private long timeForDataFlush = -1, dataFlushTimeFlex = -1;

    private AnalyticsOptions(Builder builder){

        this.storageType         = builder.storageType;
        this.timeForDataFlush    = builder.timeForDataFlush;
        this.dataFlushTimeFlex   = builder.dataFlushTimeFlex;
        this.context             = builder.context;
        this.customDeviceId      = builder.customDeviceId;

    }

    /**
     * Returns the storage type to be used.
     * @return The storage type enum.
     */
    public StorageType getStorageType()
    {
        return storageType;
    }

    /**
     * Returns the period of time between each flush, done automatically by the flush service.
     * @return The flush period time in seconds.
     */
    public long getDataFlushTime()
    {
        return timeForDataFlush;
    }

    /**
     * Returns the flex time in seconds originally set.
     * @return The flex time in seconds.
     */
    public long getDataFlushTimeFlex() {
        return dataFlushTimeFlex;
    }

    /**
     * Returns the custom device id to be used, other than the Flybits one.
     * @return The custom device id.
     */
    public String getCustomDeviceId() {
        return customDeviceId;
    }

    /**
     * Returns the {@link Context} used to initialize this {@link AnalyticsOptions}.
     * @return The context.
     */
    public Context getContext() {
        return context;
    }

    /**
     * The {@code AnalyticsOptions.Builder} class is used to set various options for the Analytics
     * SDK, and build a {@link AnalyticsOptions} object for initialization.
     */
    public static class Builder{

        private String customDeviceId;
        private StorageType storageType;
        private long timeForDataFlush = -1, dataFlushTimeFlex = -1;
        private Context context;

        private ArrayList<Class<? extends UploadChannel> > customUploadChannels;

        /**
         * Default Constructor that initializes all the necessary {@code Builder} variables.
         *
         * @param context Current application state of the application.
         */
        public Builder(@NonNull Context context){
            this.context            = context;
            this.storageType        = StorageType.SQLITE_DB;
            this.customUploadChannels = null;
            this.customDeviceId = null;
        }

        /**
         * Creates the {@link AnalyticsOptions} object using the settings that were configured.
         * @return The {@link AnalyticsOptions} object.
         */
        public AnalyticsOptions build(){
            return new AnalyticsOptions(this);
        }

        /**
         * Sets the storage type that analytic data will be saved as, before being flushed to the server.
         *
         * @param type The storage type.
         *
         * @return The {@link AnalyticsOptions.Builder} object to support chaining.
         */
        public Builder setStorageType(@NonNull StorageType type)
        {
            this.storageType = type;
            return this;
        }

        /**
         * If this is called, a service will automatically flush the saved analytic event data. The
         * amount of time between flushes is defined by timeUpdate and timeFlex.
         *
         * @param timeUpdate The amount of time per flush.
         * @param timeFlex If there is an issue flushing, this is the amount of time to wait before
         *                 trying again.
         * @param metric The time unit that the first two parameters are being represented in.
         *
         * @return The {@link AnalyticsOptions.Builder} object to support chaining.
         */
        public Builder setUploadServiceTime(long timeUpdate, long timeFlex, @NonNull TimeUnit metric)
        {

            this.timeForDataFlush = metric.toSeconds(timeUpdate);
            this.dataFlushTimeFlex = metric.toSeconds(timeFlex);

            if (timeForDataFlush < 60)
                timeForDataFlush = 60;
            if (dataFlushTimeFlex < 60)
                dataFlushTimeFlex = 60;

            return this;
        }

        /**
         * Sets a custom device id to use, instead of the one provided by Flybits.
         *
         * @param id A custom device id.
         *
         * @return The {@link AnalyticsOptions.Builder} object to support chaining.
         */
        public Builder setCustomDeviceId(String id)
        {
            customDeviceId = id;
            return this;
        }

    }
}
