package com.flybits.commons.library.models;

import android.arch.persistence.room.ColumnInfo;
import android.arch.persistence.room.Entity;
import android.arch.persistence.room.Ignore;
import android.arch.persistence.room.PrimaryKey;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;

/**
 * This class represents a {@code User} that can be found within this application environment. A
 * {@code User} is defined by their profile attributes as well an any additional metadata associated
 * to the {@code User} that may be specific to the application.
 */

@Entity(tableName = "user")
public class User implements Parcelable {

    @ColumnInfo(name = "deviceID")
    private String deviceID;

    @ColumnInfo(name = "email")
    private String email;

    @ColumnInfo(name = "firstName")
    private String firstName;

    @PrimaryKey
    @ColumnInfo(name = "id")
    @NonNull
    private String id;

    @ColumnInfo(name = "lastName")
    private String lastName;

    /**
     * Default constructor used to define a ROOM DB Entity for the {@code User} object. This
     * constructor should not be used by the application and is meant for internal use only.
     */
    public User (){}

    /**
     * Constructor used for un-flattening a {@code User} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code User} parcel.
     */
    @Ignore
    public User(Parcel in){
        id                  = in.readString();
        firstName           = in.readString();
        lastName            = in.readString();
        email               = in.readString();
        deviceID            = in.readString();
    }

    /**
     * Default constructor that creates {@code User} object.
     *
     * @param id The unique identifier of the {@code User}.
     * @param email The email of the {@code User}.
     * @param deviceID The device identifier for the currently logged in {@code User}.
     */
    @Ignore
    public User(@NonNull String id, @NonNull String email, @NonNull String deviceID){
        this.deviceID       = deviceID;
        this.email          = email;
        this.id             = id;
    }

    /**
     * Get the unique device identifier for the currently logged in {@code User}.
     *
     * @return The unique device identifier.
     */
    public String getDeviceID() {
        return deviceID;
    }

    /**
     * Get the email of the {@code User}.
     *
     * @return The unique email registered for the logged in {@code User}.
     */
    public String getEmail() {
        return email;
    }

    /**
     * Get the first name of the logged in {@code User}.
     *
     * @return The first name of the logged in {@code User}.
     */
    public String getFirstName() {
        return firstName;
    }

    /**
     * Get the unique identifier of the logged in {@code User}.
     *
     * @return The unique identifier of the logged in {@code User}.
     */
    public String getId() {
        return id;
    }

    /**
     * Get the last name of the logged in {@code User}.
     *
     * @return The last name of the logged in {@code User}.
     */
    public String getLastName() {
        return lastName;
    }

    /**
     * Set the Flybits Device identifier for this instance of the application for this device.
     *
     * @param deviceID The unique identifier of the device.
     */
    public void setDeviceID(String deviceID) {
        this.deviceID = deviceID;
    }

    /**
     * Set the email of the user logged in to the application.
     *
     * @param email The unique email for the user who is logged into the application.
     */
    public void setEmail(String email) {
        this.email = email;
    }

    /**
     * Set the first name of the user.
     *
     * @param firstName The first name of the user who is logged into the application.
     */
    public void setFirstName(String firstName) {
        this.firstName = firstName;
    }

    /**
     * Set the unique identifier for the user who is logged into the application.
     *
     * @param id The unique identifier for the user who is logged into the application.
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * Set the last name of the user.
     *
     * @param lastName The last name of the user who is logged into the system.
     */
    public void setLastName(String lastName) {
        this.lastName = lastName;
    }

    /**
     * Set the first and last name of the logged in {@code User}.
     *
     * @param firstName The first name of the logged in {@code User}.
     * @param lastName The last name of the logged in {@code User}.
     */
    public void setName(String firstName, String lastName) {
        this.firstName  = firstName;
        this.lastName   = lastName;
    }

    /**
     * Describe the kinds of special objects contained in this Parcelable's marshalled representation.
     *
     * @return a bitmask indicating the set of special object types marshalled by the Parcelable.
     */
    public int describeContents() {
        return 0;
    }

    @Override
    public boolean equals(Object o) {
        if (!(o instanceof User))
            return false;

        User another = (User) o;
        return (another.id.equals(id));
    }

    /**
     * Flatten this {@code User} into a Parcel.
     *
     * @param out The Parcel in which the {@code User} object should be written.
     * @param flags Additional flags about how the {@code User} object should be written.
     * May be 0 or {@link #PARCELABLE_WRITE_RETURN_VALUE}.
     */
    public void writeToParcel(Parcel out, int flags) {
        out.writeString(id);
        out.writeString(firstName);
        out.writeString(lastName);
        out.writeString(email);
        out.writeString(deviceID);
    }

    /**
     * Parcelable.Creator that instantiates {@code User} objects
     */
    public static final Creator<User> CREATOR = new Creator<User>(){
        public User createFromParcel(Parcel in){
            return new User(in);
        }

        public User[] newArray(int size){
            return new User[size];
        }
    };
}
