package com.flybits.commons.library.api.idps;


import android.support.annotation.NonNull;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * The {@code OAuthIDP} class is responsible for creating a specific IDentity Provider (IDP) that
 * is managed by any O-Auth based 3rd party integration such Facebook or Google. The
 * {@code OAuthIDP} creates a {@link com.flybits.commons.library.models.User} based on the
 * information retrieved from the O-Auth provider regardless of whether it is Facebook, Google, etc.
 */
public class OAuthIDP extends IDP{

    private static final String AUTHENTICATE_ENDPOINT       = "/oauth";

    private String accessToken;
    private String clientID;
    private String provider;

    /**
     * Constructor used to register an O-Auth identity provider that needs only the
     * {@code accessToken}. Currently, if you plan on supporting the Facebook or Google IDP, this
     * constructor should be used.
     *
     * @param provider The provider which indicates which IDP the {@code accessToken} and
     *                 {@code clientID} are associated to.
     * @param accessToken The {@code accessToken} that is retrieved from the 3rd party O-Auth
     *                    provider.
     */
    public OAuthIDP(@NonNull String provider, @NonNull String accessToken){
        super(provider);
        this.provider           = provider;
        this.accessToken        = accessToken;
    }

    /**
     * Constructor used to register an O-Auth identity provider that needs only the
     * {@code accessToken}. Currently, if you plan on supporting the Facebook or Google IDP, this
     * constructor should be used.
     *
     * @param idp The {@link SupportedIDP} that indicates the provider that the {@code accessToken}
     *            should be associated to.
     * @param accessToken The {@code accessToken} that is retrieved from the 3rd party O-Auth
     *                    provider.
     */
    public OAuthIDP(@NonNull SupportedIDP idp, @NonNull String accessToken) {
        this(idp.getKey(), accessToken);
    }

    /**
     * Constructor used to register an O-Auth identity provider that needs both an
     * {@code accessToken}, as well as a {@code clientID} in order to successfully connect to the
     * IDP.
     *
     * @param provider The provider which indicates which IDP the {@code accessToken} and
     *                 {@code clientID} are associated to.
     * @param accessToken The {@code accessToken} that is retrieved from the 3rd party O-Auth
     *                    provider.
     * @param clientID The {@code clientID} refers to the project that is registered with the 3rd
     *                 party O-Auth provider for this specific application instance of the SDK.
     */
    public OAuthIDP(@NonNull String provider, @NonNull String accessToken, String clientID){
        this(provider, accessToken);
        this.clientID  = clientID;
    }

    /**
     * Constructor used to register an O-Auth identity provider that needs only the
     * {@code accessToken}. Currently, if you plan on supporting the {@link SupportedIDP#FACEBOOK}
     * or {@link SupportedIDP#GOOGLE}, this constructor should be used.
     *
     * @param idp The {@link SupportedIDP} that indicates the provider that the {@code accessToken}
     *            should be associated to.
     * @param accessToken The {@code accessToken} that is retrieved from the 3rd party O-Auth
     *                    provider.
     * @param clientID The {@code clientID} refers to the project that is registered with the 3rd
     *                 party O-Auth provider for this specific application instance of the SDK.
     */
    public OAuthIDP(@NonNull SupportedIDP idp, @NonNull String accessToken, String clientID){
        this (idp.getKey(), accessToken, clientID);
    }

    @Override
    public String getAuthenticationEndPoint() {
        return AUTHENTICATE_ENDPOINT;
    }

    @Override
    public JSONObject getBody() {
        JSONObject bodyObject   = new JSONObject();

        try {
            bodyObject.put("provider", provider);
        }catch (JSONException e){}

        try {
            bodyObject.put("accessToken", accessToken);
        } catch (JSONException e) {}

        if (clientID != null) {
            try {
                bodyObject.put("clientID", clientID);
            } catch (JSONException e) {}
        }

        return bodyObject;
    }

    /**
     * This enumerator indicates which O-Auth based Identity Providers are supported by Flybits. It
     * is possible for this enumerator to be outdated depending on how often you update this SDK. If
     * the IDP that you want to use is listed here you should use the
     * {@link OAuthIDP#OAuthIDP(SupportedIDP, String)} otherwise if you believe that Flybits uses
     * accepts another IDP that is not listed here, you should use the more generic constructor
     * {@link OAuthIDP#OAuthIDP(String, String)} or {@link OAuthIDP#OAuthIDP(String, String, String)}.
     */
    public enum SupportedIDP {
        /**
         * Indicates the gender is not known.
         */
        FACEBOOK("facebook"),

        /**
         * Indicates the gender as male.
         */
        GOOGLE("gplus");

        private final String key;

        /**
         * Constructor that defines the key for each {@code SupportedIDP} option.
         *
         * @param key the string value representing each {@code SupportedIDP} option.
         */
        SupportedIDP(String key) {
            this.key = key;
        }

        /**
         * Get the string representation for the {@code SupportedIDP} option.
         *
         * @return string representation of the {@code SupportedIDP} option.
         */
        public String getKey() {
            return this.key;
        }

        /**
         * Get the {@code SupportedIDP} enum value corresponding to an String representation.
         *
         * @param key the string representation of the {@code SupportedIDP} enum.
         *
         * @return The {@code SupportedIDP} enum for the String representation.
         */
        public static SupportedIDP fromKey(String key) {
            for(SupportedIDP type : SupportedIDP.values()) {
                if(type.getKey().equalsIgnoreCase( key)) {
                    return type;
                }
            }
            return null;
        }
    }
}
