package com.flybits.commons.library.caching;

import android.arch.lifecycle.LiveData;
import android.arch.lifecycle.Observer;
import android.support.annotation.Nullable;

/**
 * The {@code FlybitsUIObjectObserver} class is responsible for creating an {@code Observer} that
 * listens to local Database changes. Once the {@code Observer} detects that there is a change in
 * data in one of the Database fields will be will notify the application using the
 * {@link DataChanged#add(DataChanged)} callback. Within this callback method the application will
 * be able to perform all the necessary UI changes.
 *
 * <p>The {@code FlybitsUIObjectObserver} is responsible for queries that are associated to a
 * single object rather than a list of objects. If you are interested in retrieving data changes to
 * a list of objects then you should look at the {@link FlybitsUIListObserver} class.</p>
 *
 * @param <T> The object that {@code Observer} is observing for content changes.
 */
public abstract class FlybitsUIObjectObserver<T> {

    private DataChanged<T> callback;
    private LiveData<T> item;
    private Observer<T> observer;

    /**
     * Default Constructor used to initialize the {@code FlybitsUIListObserver} object.
     */
    public FlybitsUIObjectObserver(){}

    /**
     * Adds a {@link DataChanged} callback that is responsible for indicating to the UI application
     * when data has changes. When data does change then the UI should make the appropriate changes.
     *
     * @param callback The {@link DataChanged} callback that is initiated when there is a change in
     *                 data through the Database.
     */
    public void add(DataChanged<T> callback){
        this.callback   = callback;
    }

    /**
     * Get the current {@code LiveData} object.
     *
     * @return The current {@code LiveData} object, if {@code setItem(LiveData)} has not been
     * called, then this method will return null.
     */
    @Nullable public LiveData<T> getData(){
        return item;
    }

    /**
     * Removes the {@code Observer} from reading changes to the object within the database. In most
     * cases, this should occur when the UI that is listening for changes is destroyed.
     */
    public void remove(){
        if (item != null && item.hasActiveObservers()){
            item.removeObserver(observer);
        }
    }

    /**
     * Sets the {@code LiveData} item that should be observed for changes.
     *
     * @param item The {@code LiveData} item that should be observed for changes.
     */
    public void setItem(LiveData<T> item){
        this.item   = item;
        this.observer   = t -> {
            if (callback != null){
                callback.onUpdate(t);
            }
        };
        this.item.observeForever(observer);
    }

    /**
     * The {@code DataChanged} is a callback responsible for indicating to the UI application when
     * data within the database, for which this class is observing, has changed.
     *
     * @param <T> The object that {@code Observer} is observing for content changes.
     */
    public interface DataChanged<T>{

        /**
         * Method responsible for indicating to the UI application that a change in the observed
         * data has changed.
         *
         * @param data The object from the database that has changed. It is possible that only some
         *             properties of the object have changed so it is up to the application to
         *             determine whether all data should be replaced or only some.
         */
        void onUpdate(@Nullable T data);
    }
}
