package com.flybits.commons.library.caching;

import android.arch.lifecycle.LiveData;
import android.arch.lifecycle.Observer;
import android.support.annotation.Nullable;

import java.util.ArrayList;
import java.util.List;

/**
 * The {@code FlybitsUIListObserver} class is responsible for creating an {@code Observer} that
 * listens to local Database changes. Once the {@code Observer} detects that there is a change in
 * data in one of the Database fields will be will notify the application using the
 * {@link DataChanged#add(DataChanged)} callback. Within this callback method the application will
 * be able to perform all the necessary UI changes.
 *
 * <p>The {@code FlybitsUIListObserver} is responsible for queries that are associated to a List of
 * object rather than a single object. If you are interested in Single object manipulation then you
 * should look at the {@link FlybitsUIObjectObserver} class.</p>
 *
 * @param <T> The object that {@code Observer} is observing for content changes.
 */
public abstract class FlybitsUIListObserver<T> {

    private DataChanged<T> callback;
    private LiveData<List<T>> item;
    private Observer<List<T>> observer;

    private List<T> itemsLoaded;

    /**
     * Default Constructor used to initialize the {@code FlybitsUIListObserver} object.
     */
    public FlybitsUIListObserver(){
        itemsLoaded = new ArrayList<>();
    }

    /**
     * Adds a {@link DataChanged} callback that is responsible for indicating to the UI application
     * when data has changes. When data does change then the UI should make the appropriate changes.
     *
     * @param callback The {@link DataChanged} callback that is initiated when there is a change in
     *                 data through the Database.
     */
    public void add(DataChanged<T> callback){
        this.callback   = callback;
    }

    /**
     * Get the current {@code LiveData} object.
     *
     * @return The current {@code LiveData} object, if {@code setItems(LiveData)}
     * has not been called, then this method will return null.
     */
    @Nullable public LiveData<List<T>> getData(){
        return item;
    }

    /**
     * Removes the {@code Observer} from reading changes to the List of objects within the database.
     * In most cases, this should occur when the UI that is listening for changes is destroyed.
     */
    public void remove(){
        if (item != null && item.hasActiveObservers()){
            item.removeObserver(observer);
        }
    }

    /**
     * Return a list of already loaded items. If no items have been loaded or there are no item
     * cached then this will return a empty list of items.
     *
     * @return A list of items that are cached and loaded.
     */
    public List<T> getItems(){
        return itemsLoaded;
    }

    /**
     * Sets the {@code LiveData} list that is being observed for changes.
     *
     * @param item The {@code LiveData} list that is be observed for changes.
     */
    public void setItems(LiveData<List<T>> item){
        this.item   = item;
        this.observer   = items -> {

            if (itemsLoaded.size() == 0){
                //First Time Load

                itemsLoaded.addAll(items);

                if (callback != null){
                    callback.onCacheLoaded(items);
                }
            }else {
                if (callback != null) {
                    callback.onUpdate(items);
                }
            }
        };
        this.item.observeForever(observer);
    }
    
    /**
     * The {@code DataChanged} is a callback responsible for indicating to the UI application when
     * data within the database, for which this class is observing, has changed.
     *
     * @param <T> The object that {@code Observer} is observing for content changes.
     */
    public interface DataChanged<T> {

        /**
         * Method responsible for indicating to the UI application that a change in the observed
         * data has changed.
         *
         * @param data The list of objects from the database. Some of these objects will have
         *             changed while other have not.
         */
        void onUpdate(List<T> data);

        /**
         * Indicates that the cache has successfully loaded.
         *
         * @param data The list of cached items.
         */
        void onCacheLoaded(List<T> data);
    }
}
