package com.flybits.internal;

import android.app.Service;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Handler;
import android.os.IBinder;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.content.LocalBroadcastManager;

import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.api.FlyJWT;
import com.flybits.commons.library.api.FlybitsAPIConstants;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.internal.Result;

import org.eclipse.paho.android.service.MqttAndroidClient;
import org.eclipse.paho.client.mqttv3.DisconnectedBufferOptions;
import org.eclipse.paho.client.mqttv3.IMqttActionListener;
import org.eclipse.paho.client.mqttv3.IMqttDeliveryToken;
import org.eclipse.paho.client.mqttv3.IMqttToken;
import org.eclipse.paho.client.mqttv3.MqttCallbackExtended;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.eclipse.paho.client.mqttv3.MqttMessage;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Random;
import java.util.Vector;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public class FlybitsMQTTService extends Service{

    static final String INTERNAL_FLYBITS_PUSH_SCOPE_KERNEL    = "com.flybits.action.push.scope.kernel";
    static final String INTERNAL_FLYBITS_PUSH_SCOPE_PUSH      = "com.flybits.action.push.scope.push";
    static final String INTERNAL_FLYBITS_PUSH_SCOPE_CONTEXT   = "com.flybits.action.push.scope.context";

    static final String INTERNAL_FLYBITS_PUSH_DATA      = "com.flybits.action.push.param.data";
    static final String INTERNAL_FLYBITS_PUSH_ACTION    = "com.flybits.action.push.param.action";
    static final String INTERNAL_FLYBITS_PUSH_ENTITY    = "com.flybits.action.push.param.entity";

    private final int MINTIMEOUT    = 1;
    private final int MAXTIMEOUT    = 256;
    private int timeout             = MINTIMEOUT;

    public static final String MQTT_SUBSCRIBE       = "mqtt-subscribe";
    public static final String MQTT_DISCONNECT      = "mqtt-disconnect";
    public static final String MQTT_CONNECT      = "mqtt-connect";
    public static final String MQTT_PUBLISH         = "mqtt-publish";

    public static final String IS_SUBSCRIBE         = "mqtt-event:IS_SUBSCRIBE";
    public static final String TOPICS               = "topics";

    private Handler handler;
    private Vector<String> topics;
    private Random rand;

    private static final int QOS_WILL               = 1;
    private static final int QOS                    = 0;
    private static final int KEEP_ALIVE             = 30;
    private static final String _TAG                = "FlybitsMQTTService";

    private MqttAndroidClient client;

    @Override
    public void onCreate() {
        super.onCreate();
    }

    @Nullable
    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {

        LocalBroadcastManager.getInstance(this).registerReceiver(mSubscription, new IntentFilter(MQTT_SUBSCRIBE));
        LocalBroadcastManager.getInstance(this).registerReceiver(mDisconnection, new IntentFilter(MQTT_DISCONNECT));
        LocalBroadcastManager.getInstance(this).registerReceiver(mPublish, new IntentFilter(MQTT_PUBLISH));
        LocalBroadcastManager.getInstance(this).registerReceiver(mConnect, new IntentFilter(MQTT_CONNECT));

        handler     = new Handler();
        topics      = new Vector<>();
        rand        = new Random();
        new Thread(new Runnable() {
            public void run() {
                connect();
            }
        }).start();
        return START_STICKY;
    }

    //region Options
    private void connect(){
        final String projectID      = SharedElements.getProjectID(getBaseContext());
        final String deviceID       = SharedElements.getDeviceID(getBaseContext());
        final String userID         = SharedElements.getUserID(getBaseContext());
        String jwt                  = SharedElements.getSavedJWTToken(getBaseContext());

        if (jwt == null){
            jwt = "";
        }

        String MQTT_url             = FlybitsAPIConstants.getMQTTURL();
        final String clientID       = deviceID+"_"+projectID;
        client                      = new MqttAndroidClient(getBaseContext(), MQTT_url, clientID);

        try {
            IMqttToken token        = client.connect(getMqttConnectionOptions(jwt, userID, deviceID, projectID));
            if (token != null){
                token.setActionCallback(new IMqttActionListener() {
                    @Override
                    public void onSuccess(IMqttToken asyncActionToken) {

                        Logger.setTag(_TAG).d("connected");

                        client.setBufferOpts(getDisconnectedBufferOptions());

                        new Thread(new Runnable() {
                            public void run() {
                                if (client.isConnected()) {

                                    topics.add(SharedElements.getDeviceID(getBaseContext()) + "_" + SharedElements.getProjectID(getBaseContext()));
                                    if (topics.size() > 0){
                                        ArrayList<String> list = new ArrayList<String>(topics);
                                        subscribe(list);
                                        topics.clear();
                                    }
                                }
                            }
                        }).start();
                    }

                    @Override
                    public void onFailure(IMqttToken asyncActionToken, Throwable exception) {

                        Logger.setTag(_TAG).d( "Failure " + exception.toString());
                        if (exception instanceof MqttException) {
                            if (((MqttException) exception).getReasonCode() == MqttException.REASON_CODE_FAILED_AUTHENTICATION) {

                                final ExecutorService executorService = Executors.newSingleThreadExecutor();
                                executorService.execute(new Runnable() {
                                    public void run() {
                                        try {
                                            Result result = FlyJWT.refreshJWT(getBaseContext());
                                            if (result.getStatus() != RequestStatus.COMPLETED){
                                                disconnect();
                                            }
                                        } catch (FlybitsException | MqttException e) {
                                            Logger.exception("FlybitsMQTTService.connect.onFailure", e);
                                        }
                                    }
                                });
                            }
                        }
                        errorConnecting();
                    }
                });
            }
        } catch (MqttException e) {
            Logger.exception("FlybitsMQTTService.onCreate", e);
        }

        client.setCallback(callback);
    }

    private void subscribe(ArrayList<String> listOfTopics){
        for (String topic : listOfTopics) {

            if (!topic.isEmpty()) {

                if (client != null && client.isConnected()) {
                    try {
                        client.subscribe(topic, QOS);
                    } catch (MqttException e) {
                        Logger.exception("FlybitsMQTTService.subscribe", e);
                    }
                } else {
                    topics.add(topic);
                }
            }
        }
    }

    private void unsubscribe(ArrayList<String> listOfTopics){
        for (String topic : listOfTopics) {

            if (!topic.isEmpty()) {

                if (client != null && client.isConnected()) {
                    try {
                        client.unsubscribe(topic);
                    } catch (MqttException e) {
                        Logger.exception("FlybitsMQTTService.subscribe", e);
                    }
                } else {
                    topics.remove(topic);
                }
            }
        }
    }

    private void disconnect() throws MqttException {
        if (client != null && client.isConnected()) {
            IMqttToken mqttToken = client.disconnect();
            mqttToken.setActionCallback(new IMqttActionListener() {
                @Override
                public void onSuccess(IMqttToken iMqttToken) {
                    Logger.setTag(_TAG).d("Successfully disconnected");
                    client.unregisterResources();
                    client.close();
                    client = null;
                }

                @Override
                public void onFailure(IMqttToken iMqttToken, Throwable throwable) {
                    Logger.setTag(_TAG).d("Failed to disconnected " + throwable.toString());
                }
            });
        }
    }
    //endregion

    //region MQTTSettings
    private MqttConnectOptions getMqttConnectionOptions(String jwt, String userID, String deviceID, String projectID) {
        final String clientID     = deviceID+"_"+projectID;
        final String will         = userID+"_"+deviceID+"_"+projectID;

        Logger.setTag(_TAG).d("JWT Username: " + clientID);
        Logger.setTag(_TAG).d("JWT Password: " + jwt);

        MqttConnectOptions mqttConnectOptions = new MqttConnectOptions();
        mqttConnectOptions.setAutomaticReconnect(true);
        mqttConnectOptions.setCleanSession(false);
        mqttConnectOptions.setUserName(clientID);
        mqttConnectOptions.setPassword(jwt.toCharArray());
        mqttConnectOptions.setKeepAliveInterval(KEEP_ALIVE);

        try {
            mqttConnectOptions.setWill("flybits.mqtt.offline", will.getBytes("UTF-8"), QOS_WILL, false);
        }catch (UnsupportedEncodingException | IllegalStateException e){
            Logger.exception("FlybitsMQTTService.connect.UnsupportedEncodingException", e);
            return null;
        }
        return mqttConnectOptions;
    }

    @NonNull
    private DisconnectedBufferOptions getDisconnectedBufferOptions() {
        DisconnectedBufferOptions disconnectedBufferOptions = new DisconnectedBufferOptions();
        disconnectedBufferOptions.setBufferEnabled(true);
        disconnectedBufferOptions.setBufferSize(100);
        disconnectedBufferOptions.setPersistBuffer(false);
        disconnectedBufferOptions.setDeleteOldestMessages(false);
        return disconnectedBufferOptions;
    }

    private MqttCallbackExtended callback   = new MqttCallbackExtended() {
        @Override
        public void connectComplete(boolean b, String s) {

        }

        @Override
        public void connectionLost(Throwable cause) {
            Logger.setTag(_TAG).d("connectionLost: " + cause);
            if (cause != null) {
                errorConnecting();
            }
        }

        @Override
        public void messageArrived(final String topic, final MqttMessage message) throws Exception {
            new Thread(new Runnable() {
                public void run() {
                    Logger.setTag(_TAG).i(topic + ":" + message.toString());
                    String pushNotification = message.toString();

                    parseNotification(pushNotification);
                }
            }).start();
        }

        @Override
        public void deliveryComplete(IMqttDeliveryToken token) {

        }
    };
    //endregion

    //region BroadcastReceivers
    private BroadcastReceiver mSubscription = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {

            final ArrayList<String> listOfTopics  = intent.getStringArrayListExtra(TOPICS);
            final boolean isSubsribe              = intent.getBooleanExtra(IS_SUBSCRIBE, false);
            Logger.setTag(_TAG).d("Topic: " + listOfTopics);

            new Thread(new Runnable() {
                public void run() {

                    if (isSubsribe) {
                        subscribe(listOfTopics);
                    }else{
                       unsubscribe(listOfTopics);
                    }
                }
            }).start();
        }
    };

    private BroadcastReceiver mDisconnection = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            Logger.setTag(_TAG).d("BroadcastReceiver mDisconnection");
            try {
                disconnect();
            } catch (MqttException e) {
                Logger.exception("FlybitsMQTTService.mDisconnection", e);
            }
        }
    };

    private BroadcastReceiver mConnect = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            Logger.setTag(_TAG).d("BroadcastReceiver mConnection");
            final ExecutorService executorService = Executors.newSingleThreadExecutor();
            executorService.execute(new Runnable() {
                public void run() {
                    connect();
                }
            });
        }
    };

    private BroadcastReceiver mPublish = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {

        }
    };
    //endregion

    private void parseNotification(String pushNotification) {
        DeserializeMQTTNotification deserializeResult = new DeserializeMQTTNotification();
        PushInternal result = deserializeResult.fromJson(pushNotification);

        Intent intentCustom = new Intent();
        switch (result.getCategory()){
            case "kernel" :
                intentCustom.setAction(INTERNAL_FLYBITS_PUSH_SCOPE_KERNEL);
                break;
            case "push" : case "custom" :
                intentCustom.setAction(INTERNAL_FLYBITS_PUSH_SCOPE_PUSH);
                break;
            case "context" :
                intentCustom.setAction(INTERNAL_FLYBITS_PUSH_SCOPE_CONTEXT);
                break;
        }

        intentCustom.putExtra(INTERNAL_FLYBITS_PUSH_DATA,   result.getData());
        intentCustom.putExtra(INTERNAL_FLYBITS_PUSH_ACTION, result.getAction());
        intentCustom.putExtra(INTERNAL_FLYBITS_PUSH_ENTITY, result.getEntity());
        LocalBroadcastManager.getInstance(getApplicationContext()).sendBroadcast(intentCustom);
    }

    private void errorConnecting(){

        Logger.setTag(_TAG).d("errorConnecting");
        handler.removeCallbacks(runnableConnection);
        if (timeout < MAXTIMEOUT) {

            int randomNumber    = rand.nextInt(timeout*2) + timeout;
            handler.postDelayed(runnableConnection, (randomNumber*1000));
            timeout             *= 2;
        }
    }

    private Runnable runnableConnection = new Runnable(){

        @Override
        public void run() {
            new Thread(new Runnable() {
                public void run() {
                    Logger.setTag(_TAG).d("runnableConnection");
                    connect();
                }
            }).start();
        }
    };

    @Override
    public void onDestroy() {
        Logger.setTag("Testing").d("onDestroy()");
        handler.removeCallbacks(runnableConnection);
        LocalBroadcastManager.getInstance(this).unregisterReceiver(mSubscription);
        LocalBroadcastManager.getInstance(this).unregisterReceiver(mDisconnection);
        LocalBroadcastManager.getInstance(this).unregisterReceiver(mPublish);
        LocalBroadcastManager.getInstance(this).unregisterReceiver(mConnect);

    }
}
