package com.flybits.commons.library.models.internal;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

/**
 * The {@link QueryParameters} class is used to define all common parameters that can be added to
 * any GET request in order to filter the results requested. For example, you may want to add paging
 * to your GET request in order to get a different set of results.
 */
public class QueryParameters {

    private long limit;
    private long offset;
    private ArrayList<String> labels;

    /**
     * Default constructor that initializes all the local variables including {@code limit} and
     * {@code offset} used for paging, and an empty {@code labels} list for filtering by label.
     */
    public QueryParameters() {
        limit   = 0;
        offset  = -1;
        labels  = new ArrayList<>();
    }

    /**
     * Get the list of query parameters that should be added to the GET request.
     *
     * @return A list of parameters that should be added to the GET query.
     */
    public Map<String, ArrayList<String>> getQueryParams(){

        Map<String, ArrayList<String>> params = new HashMap<>();

        if (limit > 0 && offset > -1) {

            ArrayList<String> limitParam = new ArrayList<>();
            limitParam.add( String.valueOf(limit));
            params.put("limit", limitParam);

            ArrayList<String> offsetParam = new ArrayList<>();
            offsetParam.add(String.valueOf(offset));
            params.put("offset", offsetParam);
        }

        if (labels.size() > 0)
        {
            ArrayList<String> labelsParam = new ArrayList<>();
            labelsParam.addAll(labels);
            params.put("labels", labelsParam);
        }

        return params;
    }

    /**
     * Adds a paging mechanism to the request based on the number of returned results wanted and
     * the offset of the next X number of results, where X is a limit.
     *
     * @param limit The maximum number of returned objects.
     * @param offset The offset of where the next X number of response objects will be returned
     *               from where X is the limit.
     */
    public void addPaging(long limit, long offset){
        this.limit  = limit;
        this.offset = offset;
    }

    /**
     * Adds a constraint to the results, requiring them to have the given labels.
     *
     * @param labels The labels to filter by.
     */
    public void setLabels(String... labels)
    {
        if (labels == null)
            return;

        this.labels.clear();
        for (String label : labels) {
            if (label != null)
                this.labels.add(label);
        }
    }

    /**
     * Adds a constraint to the results, requiring them to have the given labels.
     *
     * @param labels A list of labels to filter by.
     */
    public void setLabels(ArrayList<String> labels)
    {
        if (labels == null)
            return;

        this.labels.clear();
        for (String label : labels)
            this.labels.add(label);
    }

}
