package com.flybits.commons.library.api;

import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.v4.content.LocalBroadcastManager;

import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.analytics.AnalyticsOptions;
import com.flybits.commons.library.analytics.Properties;
import com.flybits.commons.library.api.idps.IDP;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.ConnectionResult;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ConnectionResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.deserializations.DeserializeLogin;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.exceptions.InvalidFlybitsManagerException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.models.User;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.internal.FlybitsMQTTService;
import com.flybits.internal.db.CommonsDatabase;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import static com.flybits.commons.library.SharedElements.getSavedJWTToken;
import static com.flybits.internal.FlybitsMQTTService.MQTT_DISCONNECT;

/**
 * The {@code FlybitsManager} class is responsible for defining all {@link FlybitsScope}s as well as
 * performing all {@link User} operations such as logging in, logging out, disabling user's account,
 * user forgotten password, and refreshing the logged in user's JWT.
 */
//TODO: Move PushService to Commons as it should be used by all SDKs
public class FlybitsManager {

    static final String AUTHENTICATION_API      = "/sso/auth";
    static final String DISCONNECT_ENDPOINT     = AUTHENTICATION_API+"/logout";
    static final String ME_ENDPOINT             = AUTHENTICATION_API+"/me";

    public static boolean IS_DEBUG = false;

    private AnalyticsOptions analyticsOptions;
    private Context context;
    private boolean enableAnalytics;
    private IDP idProvider;
    private boolean isDebug;
    private ArrayList<String> languageCodes;
    private Set<FlybitsScope> listOfScopes;

    /**
     * Constructor used to define the {@code FlybitsManager} based on the {@link Builder}
     * attributes.
     *
     * @param builder The {@link Builder} that defines all the attributes about the
     * {@code FlybitsManager}.
     */
    private FlybitsManager(Builder builder){
        listOfScopes                = builder.listOfScopes;
        context                     = builder.mContext;
        isDebug                     = builder.isDebug;
        enableAnalytics             = builder.enableAnalytics;
        languageCodes               = builder.languageCodes;
        idProvider                  = builder.idProvider;

        if (languageCodes.size() == 0){
            languageCodes.add("en");
        }

        IS_DEBUG    = isDebug;
        if (enableAnalytics) {
            if (builder.analyticsOptions != null) {
                analyticsOptions = builder.analyticsOptions;
            } else {
                analyticsOptions = new AnalyticsOptions.Builder(context)
                        .setStorageType(AnalyticsOptions.StorageType.SQLITE_DB)
                        .setUploadServiceTime(60, 60, TimeUnit.HOURS)
                        .build();
            }
        }
    }

    /**
     * The {@code Builder} class is responsible for setting all the attributes associated to the
     * {@link FlybitsManager}.
     */
    public static final class Builder {

        private final Set<FlybitsScope> listOfScopes;
        private Context mContext;
        private IDP idProvider;
        private boolean isDebug;
        private boolean enableAnalytics;
        private AnalyticsOptions analyticsOptions;
        private ArrayList<String> languageCodes;

        /**
         * Constructor that initializes all the objects within the {@code Builder} class.
         *
         * @param context The {@link Context} of the application.
         */
        public Builder(@NonNull Context context) {
            this.mContext       = context;
            this.listOfScopes   = new HashSet<>();
            this.languageCodes  = new ArrayList<>();
            try {
                this.languageCodes.add(Locale.getDefault().getLanguage());
            }catch (Exception e){}
        }

        /**
         * Add a {@link FlybitsScope} to the {@link FlybitsManager}. The {@link FlybitsScope}
         * defines the SDK and all the properties associated to it.
         *
         * @param scope The {@link FlybitsScope} that is associated to this application.
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder addScope(@NonNull FlybitsScope scope) {
            listOfScopes.add(scope);
            return this;
        }

        /**
         * Build the {@link FlybitsManager} object that is used for defining the SDKs that are
         * associated to the application.
         *
         * @return {@link FlybitsManager} object which can be referenced by the SDK to get specific
         * information about the SDKs defined within the application.
         * @throws InvalidFlybitsManagerException if the {@link Builder#addScope(FlybitsScope)}
         * method was not called.
         */
        public FlybitsManager build() throws InvalidFlybitsManagerException {
            checkIfFieldsSet();
            return new FlybitsManager(this);
        }

        /**
         * Enables analytics to be automatically recorded for the SDK, as well as initialize the
         * {@link Analytics} component in the event that the application would like to use it as
         * Flybits Analytics as well.
         *
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder enableAnalytics(){
            this.enableAnalytics    = true;
            return this;
        }

        /**
         * Sets a custom Analytics configuration over the default one. Create one using {@link AnalyticsOptions.Builder}.
         *
         * @param options The {@link AnalyticsOptions} that will override the default options.
         *
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder overrideAnalyticsOptions(AnalyticsOptions options) {
            this.analyticsOptions   = options;
            return enableAnalytics();
        }

        /**
         * Sets the {@link IDP} which should be used sign into the Flybits Account Manager.
         *
         * @param idp The {@link IDP} that should be used for registering contextual information to
         *            the account.
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setAccount(IDP idp){
            this.idProvider = idp;
            return this;
        }

        /**
         * Sets the Flybits Android commons SDK in Debug mode. While in debug mode all logs will be
         * displayed within the logcat of your Android Studio including any errors/exceptions.
         *
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setDebug(){
            this.isDebug    = true;
            return this;
        }

        /**
         * Sets the Language that the SDK should parse the localized values with.
         * @param languageCode The 2-letter code, such as "en", that indicates which language should
         *                     be used for parsing localized values.
         *
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setLanguage(@NonNull String languageCode){
            languageCodes.clear();
            languageCodes.add(languageCode);
            return this;
        }

        private void checkIfFieldsSet()throws InvalidFlybitsManagerException {

            Properties properties = new Properties();
            if (listOfScopes.size() == 0){
                if (Analytics.isInitialized()) {
                    Analytics.logEventFlybits(AnalyticsCommonsEvents.EVENT_MISSING_SCOPE, properties);
                }
                throw new InvalidFlybitsManagerException("You must have at least 1 scope set to create a valid FlybitsManager object");
            }

            if (languageCodes.size() != 1 && languageCodes.get(0).length() != 2){
                if (Analytics.isInitialized()) {
                    Analytics.logEventFlybits(AnalyticsCommonsEvents.EVENT_INVALID_LANGUAGE_CODE, properties);
                }
                throw new InvalidFlybitsManagerException("Your language must be a 2-letter code. Make sure you call setLanguage(String)");
            }
        }
    }

    /**
     * Indicate that the logcat should display logs from the SDK.
     */
    public static void setDebug(){
        IS_DEBUG    = true;
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the {@link FlybitsScope#onConnected(User)}
     * method.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public ConnectionResult connect(final ConnectionResultCallback callback){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ConnectionResult query = new ConnectionResult(context, callback, executorService);

        executorService.execute(new Runnable() {
            @Override
            public void run() {

                if (idProvider == null) {
                    throw new InvalidFlybitsManagerException("You must have an IDP set. This can be done through the setAccount(IDP) method");
                }

                try{

                    boolean isConnnected = !getSavedJWTToken(context).equals("");
                    if (isConnnected){

                        final Result getJWT = FlyJWT.refreshJWT(context);
                        if (getJWT.getStatus() == RequestStatus.COMPLETED){
                            handler.post(new Runnable() {
                                @Override
                                public void run() {
                                    query.setConnected();
                                }
                            });

                            if (!Analytics.isInitialized() && enableAnalytics) {
                                Analytics.initialize(analyticsOptions);
                            }
                        }else if (getJWT.getStatus() == RequestStatus.NOT_CONNECTED){
                            isConnnected = false;
                        }else{
                            handler.post(new Runnable() {
                                @Override
                                public void run() {
                                    query.setResult(getJWT);
                                }
                            });
                        }
                    }

                    if (!isConnnected){

                        String body     = idProvider.getRequestBody(context).toString();
                        String url      = AUTHENTICATION_API + idProvider.getAuthenticationEndPoint();
                        final Result<User> authenticatedUser = FlyAway.post(context, url, body, new DeserializeLogin(), "FlybitsManager.connect", User.class);

                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                query.setResult(authenticatedUser);
                            }
                        });

                        if (authenticatedUser.getStatus() == RequestStatus.COMPLETED) {

                            Intent intentStartService = new Intent(context, FlybitsMQTTService.class);
                            context.startService(intentStartService);

                            CommonsDatabase.getDatabase(context).userDao().insert(authenticatedUser.getResult());
                            SharedElements.setConnectedIDP(context, idProvider.getProvider());

                            if (!Analytics.isInitialized() && enableAnalytics) {
                                Analytics.initialize(analyticsOptions);
                                Properties properties = new Properties();
                                properties.addProperty(AnalyticsCommonsEvents.PARAM_IDP_TYPE, idProvider.getClass().getCanonicalName());
                                Analytics.logEventFlybits(AnalyticsCommonsEvents.EVENT_SDK_CONNECTED, properties);
                            }

                            for (FlybitsScope scope : listOfScopes) {
                                scope.onConnected(authenticatedUser.getResult());
                            }
                        }
                    }

                    SharedElements.setLocalization(context, languageCodes);
                }catch (final FlybitsException e){

                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            query.setFailed(e);
                        }
                    });
                }
            }
        });
        return query;
    }

    /**
     * Clears all information associated this instance of the SDK. If you use this method
     * persistence between app launches will not be maintained, therefore it is highly recommended
     * that you use the {@link #disconnect(BasicResultCallback)} method over this {@code destroy()}
     * one.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult destroy(final BasicResultCallback callback){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(context, callback, executorService);

        executorService.execute(new Runnable() {
            @Override
            public void run() {
                try{
                    if (Analytics.isInitialized()) {
                        Analytics.destroy(context);
                    }

                    String jwtToken    = SharedElements.getSavedJWTToken(context);
                    final Result deleteUser = FlyAway.delete(context, ME_ENDPOINT, "FlybitsManager.destroy",null);

                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setResult(deleteUser);
                        }
                    });

                    if (deleteUser.getStatus() == RequestStatus.COMPLETED) {

                        CommonsDatabase.getDatabase(context).userDao().delete();
                        for (FlybitsScope scope : listOfScopes) {
                            scope.onAccountDestroyed(jwtToken);
                        }
                        clearUserInformation(context);
                    }
                }catch (final FlybitsException e){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setFailed(e);
                        }
                    });
                }
            }
        });
        return request;
    }

    /**
     * The {@code disconnect} method is responsible for clearing the server sessions with Flybits.
     * In addition, once a logout is successful it will notify all the registered
     * {@link FlybitsScope}s through the {@link FlybitsScope#onDisconnected(String)} method.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult disconnect(final BasicResultCallback callback){
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(context, callback, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                try{
                    if (!Analytics.isInitialized()) {
                        Analytics.destroy(context);
                    }

                    String jwtToken                     = SharedElements.getSavedJWTToken(context);
                    final Result<String> disconnected   = FlyAway.post(context, DISCONNECT_ENDPOINT, "", null, "FlybitsManager.disconnect", null);
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setResult(disconnected);
                        }
                    });

                    if (disconnected.getStatus() == RequestStatus.COMPLETED) {

                        CommonsDatabase.getDatabase(context).userDao().delete();
                        if (Analytics.isInitialized()) {
                            Analytics.logEventFlybits(AnalyticsCommonsEvents.EVENT_SDK_DISCONNECTED, null);
                        }
                        for (FlybitsScope scope : listOfScopes) {
                            scope.onDisconnected(jwtToken);
                        }
                        clearUserInformation(context);
                    }
                }catch (final FlybitsException e){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setFailed(e);
                        }
                    });
                }
            }
        });
        return request;
    }

    /**
     * The {@code getUser} method is responsible for retrieving user information about the Flybits
     * {@link User}. The {@link User} contains basic personal attributes such as
     * {@link User#firstName} and {@link User#lastName} as well as Flybits information such as
     * {@link User#id} and {@link User#deviceID}.
     *
     * @return The {@link ObjectResult} is an object that is returned from the Server after a HTTP
     * request is made.
     */
    public static ObjectResult<User> getUser(final Context context, ObjectResultCallback<User> callback){
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<User> request = new ObjectResult<>(context, callback, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                try{
                    final Result<User> getUser = FlyAway.get(context, ME_ENDPOINT, new DeserializeLogin(), "FlybitsManager.get", User.class);
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setResult(getUser);
                        }
                    });
                }catch (final FlybitsException e){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setFailed(e);
                        }
                    });
                }
            }
        });
        return request;
    }


    /**
     * The {@code getUser} method is responsible for retrieving user information about the Flybits
     * {@link User}. The {@link User} contains basic personal attributes such as
     * {@link User#firstName} and {@link User#lastName} as well as Flybits information such as
     * {@link User#id} and {@link User#deviceID}.
     *
     * @return The {@link ObjectResult} is an object that is returned from the Server after a HTTP
     * request is made.
     */
    public static ObjectResult<User> getUser(final Context context){
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<User> request = new ObjectResult<>(context, null, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                try{

                    final Result<User> getUser = FlyAway.get(context, ME_ENDPOINT, new DeserializeLogin(), "FlybitsManager.get", User.class);
                    if (getUser.getStatus() == RequestStatus.COMPLETED) {
                        CommonsDatabase.getDatabase(context).userDao().update(getUser.getResult());
                    }
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setResult(getUser);
                        }
                    });
                }catch (final FlybitsException e){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setFailed(e);
                        }
                    });
                }
            }
        });
        return request;
    }

    /**
     * Check to see if the SDK currently has established a connection with the Flybits Server.
     *
     * @param context The Context of the application.
     * @param confirmThroughNetwork Indicates whether or not a confirm should take place with the
     *                              server. If false, this will simply check is a token is saved
     *                              locally, this token may however not be valid. If true, it will
     *                              confirm with the server that it is valid.
     * @param callback The callback that indicates whether or not the SDK is currently connected to
     *                 Flybits.
     * @return The {@link ConnectionResult} which is executing the network request (if any) to
     * determine whether or not the connection is valid.
     */
    public static ConnectionResult isConnected(@NonNull final Context context,
                                               final boolean confirmThroughNetwork,
                                               @NonNull final ConnectionResultCallback callback){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        ConnectionResult query   = new ConnectionResult(context, callback, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                if (confirmThroughNetwork && !getSavedJWTToken(context).equals("")){

                    try{
                        final Result getJWT = FlyJWT.refreshJWT(context);

                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                if (getJWT.getStatus() == RequestStatus.NOT_CONNECTED){
                                    callback.notConnected();
                                }else if (getJWT.getStatus() == RequestStatus.COMPLETED){
                                    callback.onConnected();
                                }else{
                                    callback.onException(getJWT.getException());
                                }
                            }
                        });

                    }catch (final FlybitsException e){
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                callback.onException(e);
                            }
                        });
                    }
                }else if (getSavedJWTToken(context).equals("")){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.notConnected();
                        }
                    });

                }else{
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.onConnected();
                        }
                    });
                }
            }
        });
        return query;
    }

    static void clearUserInformation(Context context) {

        Intent intentLogoutPush   = new Intent(MQTT_DISCONNECT);
        LocalBroadcastManager.getInstance(context).sendBroadcast(intentLogoutPush);

        SharedPreferences preferences = SharedElements.getPreferences(context);
        SharedPreferences.Editor editor = preferences.edit();
        editor.clear();
        editor.apply();
    }

    IDP getIDP(){
        return idProvider;
    }

    Set<FlybitsScope> getScopes(){
        return listOfScopes;
    }

    boolean isDebug(){return isDebug;}

    boolean isAnalyticsEnabled(){return enableAnalytics;}

}
