package com.flybits.commons.library.api;

import android.content.Context;
import android.content.SharedPreferences;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.util.Log;

import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.analytics.AnalyticsBundle;
import com.flybits.commons.library.analytics.AnalyticsOptions;
import com.flybits.commons.library.analytics.Properties;
import com.flybits.commons.library.analytics.UploadChannel;
import com.flybits.commons.library.analytics.services.AnalyticsUploadService;
import com.flybits.commons.library.analytics.storage.MemoryStorage;
import com.flybits.commons.library.analytics.storage.QueueStorage;
import com.flybits.commons.library.analytics.storage.sql.SqliteStorage;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.utils.Utilities;
import com.google.android.gms.gcm.GcmNetworkManager;
import com.google.android.gms.gcm.PeriodicTask;
import com.google.android.gms.gcm.Task;

import java.util.Calendar;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * The {@code Analytics} class is the access point for all Analytics based operations in Flybits. It handles
 * both the storing of logged events which the user calls to be recorded, as well as flushing all stored
 * data to the appropriate {@link UploadChannel}.
 */
public class Analytics {

    public final static String _TAG                 = "Analytics";
    public final static String PREFS_STORAGETYPE    = "fbanalytics_storagetype";
    public final static String PREFS_CUSTOMDEVICEID = "fbanalytics_customdeviceid";

    private static Analytics mInstance;

    private Context mContext;
    private Map<Integer, AnalyticsBundle> mCurrentTimedEvents;
    private QueueStorage mQueueStorage;
    private String mCustomDeviceId;

    private FlybitsUploadChannel mFlybitsUploadChannel = new FlybitsUploadChannel();

    private boolean mIsShutdown = false;
    private ExecutorService mFlushExecutorService;

    private Analytics(AnalyticsOptions options) {
        mCurrentTimedEvents = new LinkedHashMap<>();
        mContext = options.getContext();

        //Set custom device id
        SharedPreferences preferences = SharedElements.getPreferences(mContext);
        mCustomDeviceId = options.getCustomDeviceId();
        preferences.edit().putString(PREFS_CUSTOMDEVICEID, mCustomDeviceId).apply();

        //Setup storage
        if (options.getStorageType() == AnalyticsOptions.StorageType.SQLITE_DB) {
            mQueueStorage = new SqliteStorage(mContext);
            if (mQueueStorage == null) {
                Logger.setTag(_TAG).e("Error creating SQliteDB. Resorting to memory storage.");
                mQueueStorage = new MemoryStorage();
            }
        } else if (options.getStorageType() == AnalyticsOptions.StorageType.BINARY) {
            mQueueStorage = new MemoryStorage();
        } else {
            mQueueStorage = new MemoryStorage();
        }

        preferences.edit().putString(PREFS_STORAGETYPE, options.getStorageType().getValue()).apply();

        //Setup flush service
        if (options.getDataFlushTime() != -1) {
            GcmNetworkManager mGcmNetworkManager = GcmNetworkManager.getInstance(mContext);

            PeriodicTask.Builder task = new PeriodicTask.Builder()
                    .setService(AnalyticsUploadService.class)
                    .setTag("FBAnalyticsUploadService")
                    .setPersisted(true)
                    .setPeriod(options.getDataFlushTime())
                    .setFlex(options.getDataFlushTimeFlex())
                    .setRequiredNetwork(Task.NETWORK_STATE_ANY)
                    .setUpdateCurrent(true);
            mGcmNetworkManager.schedule(task.build());
        }
    }

    private ExecutorService flush(final Context context) {
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                if (mFlybitsUploadChannel == null) {
                    mFlybitsUploadChannel = new FlybitsUploadChannel();
                }

                flushNonThreaded(context, mFlybitsUploadChannel);
            }
        });

        mFlushExecutorService = executorService;
        return executorService;
    }

    private void flushNonThreaded(final Context context, final UploadChannel channel) {

        final String deviceId   = mInstance.getDeviceId();
        final String projectId  = SharedElements.getProjectID(mContext);

        if (deviceId == null || deviceId.isEmpty()){
            Logger.setTag(_TAG).e("Analytics: There was an issue getting the device id. Flush aborted!");
            return;
        }

        if (mIsShutdown) {
            return;
        }

        if (!mFlybitsUploadChannel.isAuthed()) {
            boolean isOnInit = mFlybitsUploadChannel.onInit(context);
            if (!isOnInit){
                return;
            }
        }

        if (mIsShutdown) {
            return;
        }

        if (!mFlybitsUploadChannel.isAuthed()) {
            Logger.setTag(_TAG).e("Analytics: Cannot authenticate with Flybits. Aborting!");
            return;
        }

        synchronized (mInstance.mQueueStorage) {
            AnalyticsBundle[] bundles = mQueueStorage.getAll();

            if (bundles.length == 0) {
                Logger.setTag(_TAG).i(String.format("No events to flush for channel %s", channel.getName()));
                return;
            }

            Logger.setTag(_TAG).i(String.format("Starting flush of %d events to channel `%s`", bundles.length, channel.getName()));
            boolean result = channel.onUpload(context, deviceId, projectId, bundles);

            if (result) {
                mQueueStorage.remove(bundles.length);
                Logger.setTag(_TAG).i(String.format("Flush completed successfully", bundles.length));
            } else {
                Logger.setTag(_TAG).i(String.format("Flush failed", bundles.length));
            }
        }

    }

    /**
     * Sets a flag to abort any currently running flushes to shutdown Analytics.
     */
    void instanceDestroy() {
        //Stop any flushes happening
        mIsShutdown = true;
        if (mFlushExecutorService != null && !mFlushExecutorService.isShutdown())
            mFlushExecutorService.shutdown();
    }

    private String getDeviceId() {

        if (mCustomDeviceId == null && SharedElements.getPreferences(mContext).contains(PREFS_CUSTOMDEVICEID)) {
            mCustomDeviceId = SharedElements.getPreferences(mContext).getString(PREFS_CUSTOMDEVICEID, null);
        }

        if (mCustomDeviceId != null){
            return mCustomDeviceId;
        }else if (!SharedElements.getDeviceID(mContext).equals("")){
            return SharedElements.getDeviceID(mContext);
        }else {
            return Utilities.getDeviceID(mContext);
        }
    }

    //region Static Calls
    /*
    ================
    Static API calls
    ================
     */

    /**
     * Initializes the {@link AnalyticsOptions} that are established by the application developer when
     * the application is first started.
     *
     * @param options The {@link AnalyticsOptions} that are associated to this application instance.
     */
    static void initialize(@NonNull AnalyticsOptions options){
        mInstance = new Analytics(options);
    }

    /**
     * Ends any flush threads that may be running as well as stops the periodic upload service that
     * may have been started.
     * @param context An {@link Context} object.
     */
    static void destroy(Context context) {
        try {
            //Cancel the upload service
            GcmNetworkManager gcmNetworkManager = GcmNetworkManager.getInstance(context);
            gcmNetworkManager.cancelTask("FBAnalyticsUploadService", AnalyticsUploadService.class);
        } catch (IllegalArgumentException | NullPointerException ex) {
            Logger.exception("Analytics.destroy", ex);
        }

        //Flush anything left
        if (isInitialized()) {
            mInstance.flushNonThreaded(context, mInstance.mFlybitsUploadChannel);
            mInstance.instanceDestroy();
        }
    }

    /**
     * Logs a simple event.
     *
     * @param eventName The name of the event to be logged.
     */
    public static void logEvent(@NonNull String eventName)
    {
        logEvent(eventName, null);
    }

    /**
     * Logs an event with the given {@link Properties}. It will be sent to all upload channels
     * including Flybits'.
     *
     * @param eventName The name of the event to be logged.
     * @param properties The custom {@link Properties} this event may have.
     */
    public static void logEvent(@NonNull String eventName, Properties properties) {
        if (!isInitialized()) {
            Logger.setTag(_TAG).w("Logging of Analytics could not be completed. Please make sure that you have connected to Flybits using the FlybitsManager.connect() method.");
            return;
        }

        AnalyticsBundle.Builder bundle = new AnalyticsBundle.Builder(eventName)
                .setAppProperties(properties);

        logEventInternal(eventName, bundle);
    }

    /**
     * Logs a Flybits event with the given {@link Properties}.
     *
     * @param eventName The name of the event to be logged.
     * @param properties The custom {@link Properties} this event may have.
     */
    public static void logEventFlybits(@NonNull String eventName, Properties properties){

        if (!isInitialized()) {
            Logger.setTag(_TAG).w("Logging of Analytics could not be completed. Please make sure that you have connected to Flybits using the FlybitsManager.connect() method.");
            return;
        }

        AnalyticsBundle.Builder bundle = new AnalyticsBundle.Builder(eventName)
                .setAppProperties(properties)
                .setFlybitsEvent();

        logEventInternal(eventName, bundle);
    }

    private static void logEventInternal(String eventName, AnalyticsBundle.Builder bundle){

        if (!TextUtils.isEmpty(SharedElements.getUserID(mInstance.mContext))){
            bundle.setUserID(SharedElements.getUserID(mInstance.mContext));
        }

        AnalyticsBundle analyticsBundle     = bundle.build();
        Log.d("Tesing", "userID: "+analyticsBundle.getUserID());

        synchronized (mInstance.mQueueStorage) {
            mInstance.mQueueStorage.add(analyticsBundle);
            Logger.setTag(_TAG).i(String.format("Event \"%s\" @ time \"%d\" saved to storage.", eventName, analyticsBundle.getTimestamp()));
        }

    }

    /**
     * Logs and starts a timed event with no properties. It will be sent to all upload channels including Flybits'.
     *
     * @param eventName The name of the event to be logged.
     *
     * @return A reference to the started event is returned. This is used to end the timed event.
     */
    public static int startTimedEvent(@NonNull String eventName) {
        return startTimedEvent(eventName, null);
    }

    /**
     * Logs and starts a timed event with the given {@link Properties}. It will be sent to all upload channels including Flybits'.
     *
     * @param eventName The name of the event to be logged.
     * @param properties The custom {@link Properties} this event may have.
     *
     * @return A reference to the started event is returned. This is used to end the timed event.
     */
    public static int startTimedEvent(@NonNull String eventName, Properties properties) {

        if (!isInitialized()) {
            Logger.setTag(_TAG).w("Logging of Analytics could not be completed. Please make sure that you have connected to Flybits using the FlybitsManager.connect() method.");
            return -1;
        }

        if (properties == null){
            properties = new Properties();
        }
        long timestamp = Calendar.getInstance().getTimeInMillis();

        int hash = (eventName + "|" + timestamp).hashCode();

        AnalyticsBundle.Builder bundle = new AnalyticsBundle.Builder(eventName, AnalyticsBundle.EventType.START_TIME_EVENT, String.valueOf(hash))
                .setAppProperties(properties);

        if (!TextUtils.isEmpty(SharedElements.getUserID(mInstance.mContext))){
            bundle.setUserID(SharedElements.getUserID(mInstance.mContext));
        }

        AnalyticsBundle analyticsBundle = bundle.build();
        mInstance.mCurrentTimedEvents.put(hash, analyticsBundle);

        synchronized (mInstance.mQueueStorage) {
                mInstance.mQueueStorage.add(analyticsBundle);
        }
        Logger.setTag(_TAG).i(String.format("Timed event \"%s\" started @ time \"%d\".", eventName, analyticsBundle.getTimestamp()));
        return hash;
    }

    /**
     * Logs that the timed event has ended.
     *
     * TODO: Missing @return
     * @param reference The reference to the started timed event that will end.
     */
    public static boolean endTimedEvent(int reference) {

        if (!isInitialized()) {
            Logger.setTag(_TAG).e("Logging of Analytics could not be completed. Please make sure that you have connected to Flybits using the FlybitsManager.connect() method.");
            return false;
        }

        if (mInstance.mCurrentTimedEvents.containsKey(reference)) {
            AnalyticsBundle startBundle = mInstance.mCurrentTimedEvents.get(reference);
            long endTime = Calendar.getInstance().getTimeInMillis();

            AnalyticsBundle.Builder bundle = new AnalyticsBundle.Builder(startBundle.getEventName(),
                    AnalyticsBundle.EventType.END_TIME_EVENT, String.valueOf(reference));

            if (!TextUtils.isEmpty(SharedElements.getUserID(mInstance.mContext))){
                bundle.setUserID(SharedElements.getUserID(mInstance.mContext));
            }

            AnalyticsBundle analyticsBundle = bundle.build();

            mInstance.mCurrentTimedEvents.remove(reference);
            synchronized (mInstance.mQueueStorage) {
                mInstance.mQueueStorage.add(analyticsBundle);
            }

            Logger.setTag(_TAG).i(String.format("Event \"%s\" ended at \"%d\".", startBundle.getEventName(), endTime));
            return true;
        } else {
            Logger.setTag(_TAG).w("Warning: Called `endTimedEvent` on an unknown reference id");
        }

        return false;
    }

    /**
     * Sends all queued up events stored on the phone to their appropriate channels.
     *
     * @param context An Android context.
     */
    public static ExecutorService flushEvents(@NonNull Context context) {

        if (!isInitialized()) {
            Logger.setTag(_TAG).e("Logging of Analytics could not be completed. Please make sure that you have connected to Flybits using the FlybitsManager.connect() method.");
            return null;
        }
        return mInstance.flush(context);
    }

    /**
     * Returns if the Analytics SDK is initialized or not.
     *
     * @return true if initialized.
     */
    public static boolean isInitialized() {
        return mInstance != null;
    }

    public static void networkFailure(String url, String reason, Properties properties){
        properties.addProperty(AnalyticsCommonsEvents.PARAM_API_REQUEST, url);
        properties.addProperty(AnalyticsCommonsEvents.PARAM_REASON, reason);
        Analytics.logEventFlybits(AnalyticsCommonsEvents.EVENT_SDK_ERROR_NETWORK, properties);
    }

    public static void deserializationFailure(String apiRequest, Properties properties){
        properties.addProperty(AnalyticsCommonsEvents.PARAM_API_REQUEST, apiRequest);
        Analytics.logEventFlybits(AnalyticsCommonsEvents.EVENT_SDK_ERROR_PARSING, properties);
    }
}
