package com.flybits.internal.models.preferences;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;

import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.internal.db.CommonsDatabase;
import com.flybits.internal.db.models.Preference;

import java.util.ArrayList;
import java.util.List;

/**
 * The {@code Favourite} class is responsible for providing convenience methods that allow the SDKs
 * to easily add, remove, and get favourites. In addition to saving and retrieving
 * {@code Favourites} online this class also allows you to save {@code Favourites} within a local
 * DB which can be accessed at any point.
 */
public class Favourite extends FlybitsPreferences{

    /**
     * Default Constructor used to define the context of the {@code Activity} that is calling the
     * {@code Favourites}.
     *
     * @param context The context of the activity.
     */
    public Favourite(Context context) {
        super(context, new Handler(Looper.getMainLooper()));
    }

    /**
     * Constructor used to define the context of the {@code Activity} that is calling the
     * {@code Favourites}.
     *
     * @param context The context of the activity.
     * @param handler The {@code Handler} that is used to provide the ability to run code in the
     *                main UI Thread.
     */
    public Favourite(Context context, Handler handler) {
        super(context, handler);
    }

    /**
     * Add the String representation of the value to Favourites preferences based on the provided
     * {@code key}.
     *
     * @param key The key that the {@code Favourite} should be saved under.
     * @param value The String representation of the value that should be saved. If this value is an
     *              object it would be good to serialize (ex. JSON) the object before calling this
     *              method.
     * @param callback The callback that indicates whether or not the request was successful.
     */
    public void add(final String key, final String value, final BasicResultCallback callback) {
        getExecutorService().execute(new Runnable() {
            @Override
            public void run() {
                Preference preference   = new Preference(key, value);
                CommonsDatabase.getDatabase(getContext()).preferenceDAO().insert(preference);
                List<String> listOfItemsSaved = CommonsDatabase.getDatabase(getContext()).preferenceDAO().getIdsByKey(key);
                if (callback != null && listOfItemsSaved != null && listOfItemsSaved.size() > 0){
                    Favourite.this.putStringList(key, (ArrayList<String>) listOfItemsSaved, callback);
                }
            }
        });
    }

    /**
     * Remove the String representation of the value from the Favourites preferences based on the
     * provided {@code key}.
     *
     * @param key The key that the {@code Favourite} should be removed from.
     * @param value The String representation of the value that should be removed. If this value is
     *              an object it would be good to serialize (ex. JSON) the object before calling
     *              this method.
     * @param callback The callback that indicates whether or not the request was successful.
     */
    public void remove(final String key, final String value, final BasicResultCallback callback) {
        getExecutorService().execute(new Runnable() {
            @Override
            public void run() {
                CommonsDatabase.getDatabase(getContext()).preferenceDAO().deleteByKeyAndValue(key, value);
                List<String> listOfItemsSaved = CommonsDatabase.getDatabase(getContext()).preferenceDAO().getIdsByKey(key);
                if (callback != null && listOfItemsSaved != null && listOfItemsSaved.size() > 0){
                    Favourite.this.putStringList(key, (ArrayList<String>) listOfItemsSaved, callback);
                }
            }
        });
    }

    /**
     * Check to see if the a specific {@code value} has been already saved as a {@code Favourite}.
     *
     * @param key The key that the {@code Favourite} should be checked.
     * @param value The String representation of the value that should be check to see if it is
     *              saved. If this value is an object it would be good to serialize (ex. JSON) the
     *              object before calling this method.
     * @param callback The callback that indicates whether or not the request was successful.
     */
    public void containsDB(final String key, final String value, @NonNull final ObjectResultCallback<Boolean> callback){
        getExecutorService().execute(new Runnable() {
            @Override
            public void run() {
                if (callback != null) {
                    List<Preference> items = CommonsDatabase.getDatabase(getContext()).preferenceDAO().getIdsByKeyAndValue(key, value);
                    callback.onSuccess(items.size() > 0);
                }
            }
        });
    }

    /**
     * Get the {@code Favourite} values that are associated to the {@code key} from the local DB.
     *
     * @param key The key that the {@code Favourite} should be retrieved.
     * @param callback The callback that indicates whether or not the request was successful.
     */
    public void getFromDB(final String key, @NonNull final ObjectResultCallback<List<String>> callback){
        getExecutorService().execute(new Runnable() {
            @Override
            public void run() {
                if (callback != null) {
                    List<String> items = CommonsDatabase.getDatabase(getContext()).preferenceDAO().getIdsByKey(key);
                    callback.onSuccess(items);
                }
            }
        });
    }

    /**
     * Get the {@code Favourite} values that are associated to the {@code key} from the server.
     *
     * @param key The key that the {@code Favourite} should be retrieved.
     * @param callback The callback that indicates whether or not the request was successful.
     */
    public void getFromServer(final String key, @NonNull final ObjectResultCallback<ArrayList<String>> callback){
        Favourite.this.getStringList(key, callback);
    }
}
