package com.flybits.commons.library.models;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;

import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.FlybitsAPIConstants;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.HttpDefaultClass;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.models.internal.Result;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import static com.flybits.commons.library.SharedElements.getSavedJWTToken;

/*
Defines a single Flybits Project retrieved by getProjects() in the {@code FlybitsManager} class.
 */
public class Project {

    static final String URL_SETTINGS        = "/sso/settings/configs";
    static final String PROJECT_ENDPOINT    = "/sso/auth/project";

    private String creatorId;
    private String id;
    private boolean isActive;
    private String name;
    private String subdomain;
    private String tierId;

    /**
    Instantiates a {@code Project}.
     @param id The id of this project.
     @param creatorId The id of the user that created this project.
     @param name The name of this project
     @param subdomain The subdomain to access this project's Experience Studio (IE: subdomain.flybits.com)
     @param tierId The subscription tier this project is active on.
     @param isActive True if the project is activated.
     */
    public Project(String id, String creatorId, String name, String subdomain, String tierId, boolean isActive)
    {
        this.creatorId = creatorId;
        this.id = id;
        this.isActive = isActive;
        this.name = name;
        this.subdomain = subdomain;
        this.tierId = tierId;
    }

    /**
     * Returns this project's id.
     * @return The id of this project.
     */
    public String getId()
    {
        return id;
    }

    /**
     * Returns this project's owner's id.
     * @return The id of the owner that created this project.
     */
    public String getCreatorId()
    {
        return creatorId;
    }

    /**
     * Returns this project's name.
     * @return The name of this project.
     */
    public String getName()
    {
        return name;
    }

    /**
     * Returns this project's subdomain.
     * @return The subdomain.
     */
    public String getSubdomain()
    {
        return subdomain;
    }

    /**
     * Returns the tier level of this project.
     * @return The tier level of this project.
     */
    public String getTierId()
    {
        return tierId;
    }

    /**
     * Returns if project is active.
     * @return True if project is active.
     */
    public boolean isActive()
    {
        return isActive;
    }

    /**
     * Sets the project id to another project, refreshing the current session.
     *
     * @param projectId The project id to bind to.
     * @param callback An interface to resolve successes and failures of this call.
     * @return The {@link BasicResult} is an object that is returned from the Server after a HTTP
     * request is made.
     */
    public static BasicResult bindProject(final Context context, final String projectId, final BasicResultCallback callback)
    {
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(context, callback, executorService, handler);
        executorService.execute(new Runnable() {
            @Override
            public void run() {

                try {

                    String currentJwtToken = getSavedJWTToken(context);
                    Jwt currentJwt = Jwt.decodeJWTToken(currentJwtToken);

                    if (currentJwt.isProjectBound())
                        throw new FlybitsException("There is already a project bound, the client must logout first.");

                    HashMap<String, String> headers = new HashMap<>();
                    String url = FlybitsAPIConstants.constructGatewayURL(context, String.format("%s?projectId=%s", PROJECT_ENDPOINT, projectId));

                    try {
                        final Result bound = new HttpDefaultClass.Builder(context, true, url)
                                .addHeaders(headers)
                                .get()
                                .getResponse();
                        request.setResult(bound);
                    } catch (Exception e) {
                        throw new FlybitsException("Error Connecting to Flybits Server: FlybitsManager.bindProject");
                    }

                } catch (final FlybitsException e) {
                    request.setFailed(e);
                }
            }
        });
        return request;
    }

    /**
     * Get all the settings for a Project. These settings are currently only available through APIs
     * but will be added to the Developer Portal shortly. TODO: Change this only its on the DP.
     *
     * @param context The context of the activity that is calling this method.
     * @param clazz The Settings class that implements {@link JsonParser} so that Strings can be
     *              serialized and deserialized more easily.
     * @param callback Indicates where or not the network request was successful.
     * @param <T> The class that implements {@link JsonParser}.
     * @return The {@link ObjectResult} which contains information about network request.
     */
    public static <T extends JsonParser> ObjectResult<T> getSettings(final Context context, final Class<T> clazz, ObjectResultCallback<T> callback){
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<T> request = new ObjectResult<>(context, callback, executorService, handler);
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    String url  = URL_SETTINGS + "/project-settings";
                    final Result<T> getSettings = FlyAway.get(context, url, null, "Project.GetSettings", null);
                    //Check to make sure that there is properties that should be returned
                    if (getSettings.getStatus() == RequestStatus.COMPLETED){

                        //Parse the returned object based on the fromJSON(...) method
                        String response = getSettings.getResultAsString();
                        try{
                            JSONObject object = new JSONObject(response);
                            if (!object.isNull("settings")){
                                JSONObject settingsObject  = object.getJSONObject("settings");
                                final T newInstance    = clazz.newInstance();
                                newInstance.fromJson(settingsObject.toString());
                                request.setSuccess(newInstance);
                            }else{
                                request.setFailed(new FlybitsException("Unable to parse the JSON based on the defined fromJSON(String) method"));
                            }
                        }catch (ClassCastException | JSONException | InstantiationException | IllegalAccessException ex){
                            request.setFailed(new FlybitsException("Unable to parse the JSON based on the defined fromJSON(String) method"));
                        }
                    }else {
                        request.setResult(getSettings);
                    }
                } catch (final FlybitsException e) {
                    request.setFailed(e);
                }
            }
        });
        return request;
    }
}
