package com.flybits.commons.library.api.idps;


import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;

import com.flybits.commons.library.api.FlybitsAPIConstants;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.HttpDefaultClass;
import com.flybits.commons.library.models.internal.Result;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * The {@code FlybitsIDP} class is responsible for creating a specific IDentity Provider (IDP) that
 * is managed by Flybits. The Flybits IDP creates a {@link com.flybits.commons.library.models.User}
 * based on their First and Last name as well as an email and password. Once the user is registered
 * they will have to confirm through an email that is sent to their entered email address.
 */
//TODO: Create an AuthenticatorAccount on the Android Mobile Client which can be used to save credentials
public class FlybitsIDP extends IDP{

    private static final String AUTHENTICATE_ENDPOINT       = "/authenticate";
    private static final String RESET_PASSWORD_ENDPOINT     = "/sso/auth/sendResetPasswordEmail";

    private String firstName;
    private String lastName;
    private String email;
    private String password;

    /**
     * Constructor used to register a person to the Flybits IDP.
     *
     * @param email The unique email that is used to associated to a specific
     *              {@link com.flybits.commons.library.models.User} within the Flybits ecosystem.
     * @param password The password that is used to log the Flybits
     *                 {@link com.flybits.commons.library.models.User} into the Flybits ecosystem.
     * @param firstName The first name of the {@link com.flybits.commons.library.models.User}.
     * @param lastName The last name of the {@link com.flybits.commons.library.models.User}.
     */
    public FlybitsIDP(@NonNull String email, @NonNull String password, String firstName, String lastName){
        this (email, password);
        this.firstName  = firstName;
        this.lastName   = lastName;
    }

    /**
     * Constructor used to login a person to the Flybits IDP.
     *
     * @param email The unique email that is used to associated to a specific
     *              {@link com.flybits.commons.library.models.User} within the Flybits ecosystem.
     * @param password The password that is used to log the Flybits
     *                 {@link com.flybits.commons.library.models.User} into the Flybits ecosystem.
     */
    public FlybitsIDP(@NonNull String email, @NonNull String password){
        this ();
        this.email      = email;
        this.password   = password;
    }

    private FlybitsIDP(){
        super("flybits");
    }

    @Override
    public String getAuthenticationEndPoint() {
        return AUTHENTICATE_ENDPOINT;
    }

    @Override
    public JSONObject getBody() {
        JSONObject bodyObject   = new JSONObject();

        try {
            bodyObject.put("email", email);
        }catch (JSONException e){}
        try {
            bodyObject.put("password", password);
        }catch (JSONException e){}

        if (firstName != null && lastName != null) {
            try {
                bodyObject.put("firstName", firstName);
            } catch (JSONException e) {}

            try {
                bodyObject.put("lastName", lastName);
            } catch (JSONException e) {}
        }

        return bodyObject;
    }

    /**
     * Method used to request a new password for a {@link com.flybits.commons.library.models.User}
     * who has signed up to the Flybits IDP.
     *
     * @param context The current state of the application.
     * @param email The email that was used to register a
     *              {@link com.flybits.commons.library.models.User} to the Flybits IDP.
     * @param callback The {@link BasicResultCallback} that indicates whether or not the request was
     *                 successful.
     *
     * @return {@link BasicResult} that represents the network request for this query.
     */
    public static BasicResult requestNewPassword(@NonNull final Context context,
                                                 @NonNull final String email,
                                                 @NonNull final BasicResultCallback callback) {
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(context, callback, executorService);
        executorService.execute(new Runnable() {
            public void run() {

                String body     = "{\"email\" : \""+email+ "\"}";
                String url      = FlybitsAPIConstants.getGatewayURL() + RESET_PASSWORD_ENDPOINT;
                try{

                    final Result result = new HttpDefaultClass.Builder(context, false, url)
                            .post(body).getResponse();
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setResult(result);
                        }
                    });
                }catch (final Exception e) {
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            if (e instanceof FlybitsException) {
                                request.setFailed((FlybitsException) e);
                            } else {
                                request.setFailed(new FlybitsException(e));
                            }
                        }
                    });
                }
            }
        });
        return request;
    }

}
