package com.flybits.concierge.smartrewards.viewproviders

import android.app.Activity
import android.content.Context
import android.os.Handler
import android.support.design.widget.Snackbar
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.ImageView
import android.widget.TextView
import com.bumptech.glide.Glide
import com.bumptech.glide.request.RequestOptions
import com.flybits.android.kernel.ContentAnalytics
import com.flybits.concierge.FlybitsNavigator
import com.flybits.concierge.FlybitsViewProvider
import com.flybits.concierge.smartrewards.InternalPreferences
import com.flybits.concierge.smartrewards.R
import com.flybits.concierge.smartrewards.model.OptIn
import com.flybits.concierge.smartrewards.model.OptInInteraction
import com.flybits.concierge.smartrewards.views.RoundedCornerBorderButton
import com.flybits.concierge.viewholders.FlybitsViewHolder

/**
 * Provides UI and business logic for the [OptIn] content instance.
 *
 * Use `OptInViewProvider(Context)` to instantiate.
 *
 */
class OptInViewProvider internal constructor(private val contentAnalytics: ContentAnalytics?): FlybitsViewProvider <OptIn> {

    /**
     * Main constructor to be used for instantiating the [OptInViewProvider].
     *
     * @param context The application's [Context].
     */
    constructor(context: Context): this(ContentAnalytics(context))

    @Deprecated("A new constructor has been made available which will allow for analytics support. deprecated in version 3.0.0, will be removed in version 4.0.0", ReplaceWith("OptInViewProvider(applicationContext)"))
    constructor(): this(null)


    override fun getClassType(): Class<OptIn> {
        return OptIn::class.java
    }

    override fun getContentType(): String {
        return "ctn-flybits-optin-card"
    }

    override fun onCreateViewHolder(parent: ViewGroup): FlybitsViewHolder<OptIn> {
        return OptInViewHolder(LayoutInflater.from(parent.context).inflate(R.layout.smartrewards_item_template_reward, parent, false), contentAnalytics)
    }

    override fun onPushOpened(model: OptIn, flybitsNavigator: FlybitsNavigator) {

    }

    // View Holder for the Benefit Card
    class OptInViewHolder (val view: View, val contentAnalytics: ContentAnalytics?) : FlybitsViewHolder<OptIn>(view) {

        private val headerView: TextView = view.findViewById(R.id.smartrewards_item_template_reward_header)
        private val titleView: TextView = view.findViewById(R.id.smartrewards_item_template_reward_title)
        private val descriptionView: TextView = view.findViewById(R.id.smartrewards_item_template_reward_description)
        private val iconView: ImageView = view.findViewById(R.id.smartrewards_item_template_reward_icon)
        private val secondaryBtn: TextView = view.findViewById(R.id.smartrewards_item_template_reward_secondary_btn)
        private val primaryBtn: TextView = view.findViewById(R.id.smartrewards_item_template_reward_primary_btn)
        private val resultContainerView: View = view.findViewById(R.id.smartrewards_item_template_reward_result_container)
        private val resultStatusTextView: TextView = view.findViewById(R.id.smartrewards_item_template_reward_result_status)
        private val resultBtn: RoundedCornerBorderButton = view.findViewById(R.id.smartrewards_item_template_reward_result_btn)
        private val gradientView: View = view.findViewById(R.id.smartrewards_item_template_reward_gradient)
        private val backgroundView: ImageView = view.findViewById(R.id.smartrewards_item_template_reward_background_image)

        private val ctaContainerView: View = view.findViewById(R.id.smartrewards_item_template_reward_cta_container)

        private val contentView: View = (view.context as Activity).findViewById<ViewGroup>(android.R.id.content).getChildAt(0)

        override fun bindData(data: OptIn, flybitsNavigator: FlybitsNavigator) {

            resultBtn.visibility = View.GONE

            data.backgroundColor?.let {
                gradientView.setBackgroundColor(data.getBackgroundColor())
            }

            data.gradientDrawable?.let {
                gradientView.background = it
            }

            headerView.text = data.header.value
            titleView.text = data.title.value
            descriptionView.text = data.explanation.value

            Glide.with(iconView.context).load(data.icon).into(iconView)

            gradientView.alpha = 1.0f
            changeStatus((InternalPreferences.isContentEngaged(view.context, data.content.id)))

            data.backgroundImage?.let {
                backgroundView.visibility = View.VISIBLE
                gradientView.alpha = 0.6f
                Glide.with(view.context).load(it).apply(RequestOptions().centerCrop()).into(backgroundView)
            }

            primaryBtn.visibility = View.INVISIBLE
            if (data.primaryButton.label != null) {
                primaryBtn.visibility = View.VISIBLE
                primaryBtn.text = data.primaryButton.label.value
                primaryBtn.setBackgroundColor(data.primaryButton.getColor())
                primaryBtn.setOnClickListener {
                    contentAnalytics?.trackEngaged(data.content)
                    contentAnalytics?.trackFulfilled(data.content)  // Full-filled analytics when clicked Accept.
                    changeStatus(OptInInteraction.ENGAGED_ACCEPTED)

                    val snackbar : Snackbar = Snackbar.make(contentView, data.primaryButton.label.value, Snackbar.LENGTH_SHORT)
                    snackbar.setAction(R.string.smartrewards_undo) {
                        changeStatus(OptInInteraction.NOT_ENGAGED)
                    }
                    snackbar.addCallback(SnackbarButtonCallback(data.content.id))
                    snackbar.show()
                }
            }

            secondaryBtn.visibility = View.INVISIBLE
            if (data.secondaryButton.label != null) {
                secondaryBtn.visibility = View.VISIBLE
                secondaryBtn.text = data.secondaryButton.label.value
                secondaryBtn.setOnClickListener {
                    contentAnalytics?.trackEngaged(data.content)
                    changeStatus(OptInInteraction.ENGAGED_DECLINED)

                    val snackbar : Snackbar = Snackbar.make(contentView, data.secondaryButton.label.value, Snackbar.LENGTH_SHORT)
                    snackbar.setAction(R.string.smartrewards_undo) {
                        changeStatus(OptInInteraction.NOT_ENGAGED)
                    }
                    snackbar.addCallback(SnackbarButtonCallback(data.content.id, false, false))
                    snackbar.show()
                }
            }
        }

        // change the view based on how the user interacts with it (past and present)
        private fun changeStatus(engaged : OptInInteraction) {

            if(engaged != OptInInteraction.NOT_ENGAGED) {

                resultContainerView.visibility = View.VISIBLE
                ctaContainerView.visibility = View.GONE
                resultStatusTextView.text = if (engaged == OptInInteraction.ENGAGED_ACCEPTED)
                    view.context.getString(R.string.smartrewards_item_template_reward_pending)
                else view.context.getString(R.string.smartrewards_item_template_reward_reject)
            } else {
                resultContainerView.visibility = View.GONE
                ctaContainerView.visibility = View.VISIBLE
            }
        }
    }

    class SnackbarButtonCallback(private val contentId: String, private val track: Boolean = true, private val affirmAction : Boolean = true) : Snackbar.Callback() {
        override fun onDismissed(snackbar: Snackbar?, event: Int) {

            // with every add there must be a remove.
            snackbar?.removeCallback(this)

            if (event != DISMISS_EVENT_ACTION) { // this dismiss is due to a non-undo action and should confirm the action
                snackbar?.let { // using a let because we need the context from it

                    val optInInteraction = if (affirmAction) OptInInteraction.ENGAGED_ACCEPTED else OptInInteraction.ENGAGED_DECLINED

                    InternalPreferences.engageContent(it.context, contentId, optInInteraction)
                    if (track) {
                        val analytics = ContentAnalytics(it.context)
                        analytics.trackEngaged(contentId)

                        // wait before sending the new events up to the server
                        Handler().postDelayed({
                            analytics.flush(null, true)
                        },500)
                    }
                }
            }
        }
    }
}