package com.flybits.concierge.smartrewards.viewproviders

import android.content.Context
import android.support.v7.widget.LinearLayoutManager
import android.support.v7.widget.PagerSnapHelper
import android.support.v7.widget.RecyclerView
import android.text.TextUtils
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.ImageView
import android.widget.TextView
import com.bumptech.glide.Glide
import com.flybits.android.kernel.ContentAnalytics
import com.flybits.concierge.FlybitsNavigator
import com.flybits.concierge.FlybitsViewProvider
import com.flybits.concierge.UserActionHandler
import com.flybits.concierge.smartrewards.R
import com.flybits.concierge.smartrewards.model.Benefit
import com.flybits.concierge.smartrewards.model.Benefits
import com.flybits.concierge.smartrewards.useractionhandlers.BenefitUserActionHandler
import com.flybits.concierge.smartrewards.utils.ViewHelper
import com.flybits.concierge.smartrewards.views.RoundedCornerBorderButton
import com.flybits.concierge.viewholders.FlybitsViewHolder

/**
 * Provides UI and business logic for the [Benefits] content template.
 *
 * Use `BenefitsViewProvider(Context)` to instantiate.
 */
class BenefitsViewProvider internal constructor(private val contentAnalytics: ContentAnalytics?
                                                , val userActionHandler: UserActionHandler<Benefit>) : FlybitsViewProvider<Benefits> {

    /**
     *
     * Constructor to be used for instantiating the [BenefitsViewProvider]. Default [UserActionHandler] will be used.
     *
     * @param context The application's [Context].
     *
     */
    constructor(context: Context): this(ContentAnalytics(context), BenefitUserActionHandler())

    /**
     *
     * Constructor to be used for instantiating the [BenefitsViewProvider] with a specific [UserActionHandler].
     *
     * @param context The application's [Context].
     * @param userActionHandler [UserActionHandler] to be used in handling user actions for this content-type.
     *
     */
    constructor(context: Context, userActionHandler: UserActionHandler<Benefit>): this(ContentAnalytics(context), userActionHandler)

    @Deprecated("A new constructor has been made available which will allow for analytics support. deprecated in version 3.0.0, will be removed in version 4.0.0", ReplaceWith("BenefitsViewProvider(applicationContext)"))
    constructor(): this(null, BenefitUserActionHandler())

    override fun getClassType(): Class<Benefits> {
        return Benefits::class.java
    }

    override fun getContentType(): String {
        return "ctn-flybits-benefits-card"
    }

    override fun onCreateViewHolder(parent: ViewGroup): FlybitsViewHolder<Benefits> {
        val view = LayoutInflater.from(parent.context).inflate(R.layout.smartrewards_item_template_benefits, parent, false)
        return BenefitsViewHolder(view, contentAnalytics, userActionHandler)
    }

    override fun onPushOpened(model: Benefits, flybitsNavigator: FlybitsNavigator) {
    }

    companion object {
        private fun configureView(data: Benefit, benefitView: View, contentAnalytics: ContentAnalytics?
                                  , flybitsNavigator: FlybitsNavigator, userActionHandler: UserActionHandler<Benefit>) {
            val gradientView: View = benefitView.findViewById(R.id.smartrewards_benefit_gradient)
            val headerTextView: TextView = benefitView.findViewById(R.id.smartrewards_benefit_header)
            val titleTextView: TextView = benefitView.findViewById(R.id.smartrewards_benefit_title)
            val descriptionTextView: TextView = benefitView.findViewById(R.id.smartrewards_benefit_description)
            val sourceTextView: TextView = benefitView.findViewById(R.id.smartrewards_benefit_source)
            val sourceImageView: ImageView = benefitView.findViewById(R.id.smartrewards_benefit_sourceImg)
            val primaryButton: RoundedCornerBorderButton = benefitView.findViewById(R.id.smartrewards_benefit_primaryBtn)

            gradientView.background = data.gradientDrawable

            if (!TextUtils.isEmpty(data.header)) {
                headerTextView.visibility = View.VISIBLE
                headerTextView.text = data.header
            } else {
                headerTextView.visibility = View.GONE
            }

            titleTextView.text = data.title.value // required

            descriptionTextView.text = data.explanation.value

            primaryButton.visibility = View.INVISIBLE
            if (data.primaryButton != null
                && data.primaryButton.label != null
                && !TextUtils.isEmpty(data.primaryButton.label.value)
                && !TextUtils.isEmpty(data.primaryButton.actionURL)) {
                primaryButton.visibility = View.VISIBLE

                primaryButton.setText(data.primaryButton.label.value)
                primaryButton.setColor(data.primaryButton.getColor())

                primaryButton.setOnClickListener {
                    userActionHandler.onUserAction(BenefitUserActionHandler.TAPPED_LEARN_MORE, data, flybitsNavigator)
                    contentAnalytics?.trackEngaged(data.content)
                    contentAnalytics?.trackFulfilled(data.content) // Full-filled analytics when clicked Accept.
                }
            }

            sourceTextView.text = data.source

            if (!TextUtils.isEmpty(data.sourceImage)) {
                sourceImageView.visibility = View.VISIBLE
                Glide.with(sourceImageView.context)
                    .load(data.sourceImage)
                    .into(sourceImageView)
            } else {
                sourceImageView.visibility = View.GONE
            }
        }
    }

    // View Holder for the Benefit Card
    class BenefitsViewHolder (val view: View, val contentAnalytics: ContentAnalytics?, val userActionHandler: UserActionHandler<Benefit>) : FlybitsViewHolder<Benefits>(view) {
        private val recycler : RecyclerView = view.findViewById(R.id.smartrewards_item_template_benefits_recyclerView)
        private val single : View = view.findViewById(R.id.smartrewards_item_template_benefits_single)
        private val emptyView : View = view.findViewById(R.id.smartrewards_item_template_benefits_emptyTxt)

        override fun bindData(data: Benefits, flybitsNavigator: FlybitsNavigator) {

            data.benefits.forEach { it.content = data.content } //Set child content to parent content

            recycler.visibility = View.GONE
            single.visibility = View.GONE
            emptyView.visibility = View.GONE

            when (data.benefits.size) {
                0 -> {
                    emptyView.visibility = View.VISIBLE
                }
                1 -> { // single view
                    single.visibility = View.VISIBLE
                    configureView(data.benefits[0], view, contentAnalytics, flybitsNavigator, userActionHandler)
                }
                else -> { // multi view
                    recycler.visibility = View.VISIBLE
                    recycler.layoutManager = LinearLayoutManager(recycler.context, LinearLayoutManager.HORIZONTAL, false)
                    recycler.adapter = BenefitRecyclerAdapter(data.benefits, contentAnalytics, flybitsNavigator, userActionHandler)
                    recycler.onFlingListener = null // need to unset the fling listener. platform bug
                    val snapHelper = PagerSnapHelper()
                    snapHelper.attachToRecyclerView(recycler)
                }
            }
        }
    }

    class BenefitRecyclerAdapter(val benefits: List<Benefit>
                                 , val contentAnalytics: ContentAnalytics?
                                 , val flybitsNavigator: FlybitsNavigator
                                 , val userActionHandler: UserActionHandler<Benefit>) : RecyclerView.Adapter<BenefitRecyclerAdapter.ViewHolder>() {

        class ViewHolder(val view: View) : RecyclerView.ViewHolder (view) {
            init {
                view.layoutParams.width = ViewHelper.cardPeekWidth(view.context)
            }
        }

        override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): ViewHolder {
            return ViewHolder(LayoutInflater.from(parent.context).inflate(R.layout.smartrewards_item_template_benefit, parent, false))
        }

        override fun getItemCount(): Int {
            return benefits.size
        }

        override fun onBindViewHolder(holder: ViewHolder, position: Int) {
            configureView(benefits[position], holder.view, contentAnalytics, flybitsNavigator, userActionHandler)
        }
    }
}