package com.flybits.commons.library.utils.filters;

import android.content.Context;

import com.flybits.commons.library.api.FlybitsAPIConstants;

/**
 * The {@code LoginOptions} class is used to apply a specific type of login method when calling the
 * {@code login(LoginOptions, IRequestCallback)} SDK method. The class allows application
 * developers to define additional options that will be included as part of their Login process
 * including device identifiers, gcm tokens, OS version numbers, etc.
 *
 * @since 1.0.0
 */
public class LoginOptions {

    /**
     * Indicates a Facebook provider for the Access token.
     */
    public final static String PROVIDER_FACEBOOK        = "facebook";

    /**
     * Indicates a Flybits provider for the Access token.
     */
    public final static String PROVIDER_FLYBITS         = "flybits";

    /**
     * Enumerator used to define which method the application is attempting to use in order to login
     * to the Flybits Core.
     */
    public enum FilterBy {

        /**
         * Indicates that the application is attempting to login through the OAuth login method.
         */
        LOGIN_WITH_OAUTH,

        /**
         * Indicates that the application is attempting to login through a synthetic identifier.
         */
        LOGIN_WITH_OTHER,

        /**
         * Indicates that the application is attempting to login through the remember me token saved
         * in the SDK.
         */
        LOGIN_WITH_REMEMBER_ME_TOKEN,

        /**
         * Indicates that the application is attempting to login using the Flybits
         * username/password.
         */
        LOGIN_WITH_USERNAME,

        /**
         * Indicates that the application is attempting to login to Flybits anonymously by
         * registering first.
         */
        LOGIN_ANONYMOUSLY
    }

    /**
     *	Indicates the various 3rd party alternatives that can be used for logging a user into the
     *  Flybits system.
     */
    public enum SocialMediaType {

        /**
         * Login via Facebook's access token.
         */
        FACEBOOK
    }

    private FilterBy type;
    private String username;
    private String password;
    private String accessToken;
    private SocialMediaType socialMediaType;
    private String accessTokenSignature;
    private String dataProvider;
    private boolean retrieveRememberMeToken;
    private boolean sdkInt;
    private boolean isJWTSet;
    private String urlEndpoint;

    private LoginOptions(Builder.Options options){

        type = options.type;

        switch (type){
            case LOGIN_WITH_OAUTH:
                this.socialMediaType    = options.socialMediaType;
                this.accessToken        = options.accessToken;
                break;
            case LOGIN_WITH_REMEMBER_ME_TOKEN:
                this.accessToken            = options.accessToken;
                this.dataProvider           = options.dataProvider;
                break;
            case LOGIN_WITH_USERNAME:
                this.username   = options.username;
                this.password   = options.password;
                break;
            case LOGIN_WITH_OTHER:
                this.accessTokenSignature   = options.accessTokenSignature;
                this.accessToken            = options.accessToken;
                this.dataProvider           = options.dataProvider;
                break;
        }

        this.urlEndpoint                = options.urlEndpoint;
        this.sdkInt                     = options.sdkInt;
        this.retrieveRememberMeToken    = options.retrieveRememberMeToken;
        this.isJWTSet                   = options.isJWTSet;
    }

    /**
     * Gets the endpoint of the SDK request. This is important because it indicates which API should
     * be accessed by the SDK.
     *
     * @return The string representation of the endpoint.
     */
    public String getEndpoint(){
        return urlEndpoint;
    }

    /**
     * Get the parameter query string used to define the JSON body of the Login GET Request.
     *
     * @return The String representation of the raw JSON body that needs to be attached to the Login
     * GET Request based on the {@link LoginOptions.FilterBy} Type.
     */
    public String getFilter(){

        String rawJson = "{";
        switch (type){

            case LOGIN_WITH_OAUTH:

                switch (socialMediaType){
                    case FACEBOOK:
                        dataProvider = PROVIDER_FACEBOOK;
                        break;
                    default:
                        dataProvider = PROVIDER_FACEBOOK;
                }

                rawJson +=  "\"accessToken\": \""+accessToken + "\"," +
                            "\"providerName\": \""+ dataProvider + "\"";
                break;
            case LOGIN_WITH_REMEMBER_ME_TOKEN:

                rawJson +=  "\"accessToken\": \""+accessToken + "\"," +
                            "\"providerName\": \"" + dataProvider + "\"";
                break;
            case LOGIN_WITH_USERNAME:

                rawJson +=  "\"email\": \""+username+ "\"," +
                        "\"password\": \""+ password + "\"";
                break;
           case LOGIN_WITH_OTHER:

                rawJson +=  "\"accessToken\": \""+accessToken+ "\", " +
                            "\"accessTokenSignature\": \""+ accessTokenSignature + "\"," +
                            "\"providerName\": \""+ dataProvider + "\"";
                break;
            default:
                return null;
        }

        if (isJWTSet()){
            rawJson += (", \"includeCredentialsJwt\" : true");
        }

        if (isRememberMeSet()) {
            rawJson += (", \"rememberMe\" : true");
        }

        if (sdkInt){
            rawJson += (", \"deviceOsVersion\": \""+ android.os.Build.VERSION.SDK_INT + "\"");
        }

        rawJson += "}";
        return rawJson;
    }

    /**
     * Get the {@link FilterBy} type that indicates through which method the user is attempting to
     * login to Flybits.
     *
     * @return The {@link FilterBy} type indicating the login type.
     */

    public FilterBy getType(){
        return type;
    }

    /**
     * Indicates whether or not the JWT token should be set with the login request.
     *
     * @return true if the JWT should be set, false otherwise.
     */
    public boolean isJWTSet(){
        return isJWTSet;
    }

    /**
     * Indicates whether or not the RememberMe token should be set with the login request.
     *
     * @return true if the RememberMe token should be set, false otherwise.
     */
    public boolean isRememberMeSet(){
        return retrieveRememberMeToken;
    }

    /**
     * The {@code Options} class defines which login method the application is attempting to login
     * through. There are currently 2 options:
     * <ul>
     *     <li>{@link Options#loginViaEmail(String, String)} : Login via username/password.</li>
     *     <li>{@link Options#loginViaSyntheticID(String, String, String)} : Login via synthetic identifier.</li>
     * </ul>
     */
    public static class Builder{

        private String username;
        private String password;
        private String accessToken;
        private String accessTokenSignature;
        private String dataProvider;
        private SocialMediaType socialMediaType;
        private FilterBy type;
        private String urlEndpoint;

        /**
         * Constructor used to create the initialize the {@code Options} object without any login
         * method defined.
         */
        public Builder(){}

        /**
         * Constructor used to create the initialize the {@code Options} object without any login
         * method defined.
         *
         * @param context The context of the application.
         * @deprecated This method as been deprecated in favour of the more optimized
         * {@link Builder#Builder()}.
         */
        @Deprecated
        public Builder(Context context){}

        /**
         * Login mechanism for logging the user into the Flybits Core using a OAuth token obtained
         * from a 3rd party source such as Facebook.
         *
         * @param accessToken The access token obtained from the 3rd party source.
         * @param socialMediaType The {@link LoginOptions.SocialMediaType}
         *                        that indicates which 3rd party source the {@code accessToken} was
         *                        obtained from.
         * @return A {@link LoginOptions.Options.Builder} object
         * where additional login settings can be set.
         */
/*
        public Builder loginViaOAuth(String accessToken, SocialMediaType socialMediaType){

            this.socialMediaType = socialMediaType;
            this.accessToken = accessToken;
            type = FilterBy.LOGIN_WITH_OAUTH;

            return new Builder(this);
        }
*/
        /**
         * Login mechanism for logging the user into the Flybits Core using the saved remember Me
         * token.
         *
         * @return A {@link LoginOptions.Options.Builder} object
         * where additional login settings can be set.
         * @throws RememberMeTokenMissing If the remember me token has not been previously set.
         */
/*
        public Builder loginViaRememberMe() throws RememberMeTokenMissing{

            SharedPreferences preferences = Flybits.include(context).getFlybitsOptions().getPreferences(context);
            final String rememberMeToken = preferences.getString("rememberMeToken", "");
            if (rememberMeToken != null && rememberMeToken.length() < 1){
                throw new RememberMeTokenMissing("Remember Me Token Not Set");
            }

            type = FilterBy.LOGIN_WITH_REMEMBER_ME_TOKEN;
            this.accessToken    = rememberMeToken;
            this.dataProvider   = PROVIDER_FLYBITS;
            return new Builder(this);
        }
*/

        /**
         * Login mechanism for logging the user into the Flybits Core using a synthetic identifier
         * obtained from a 3rd party source.
         *
         * @param accessToken The access token obtained from the 3rd party source.
         * @param accessSignature The access token signature used to validate the access token.
         * @param dataProvider The provider of both the {@code accessToken} and {@code accessSignature}.
         * @return A {@link LoginOptions.Builder.Options} object
         * where additional login settings can be set.
         */
        public Options loginViaSyntheticID(String accessToken, String accessSignature, String dataProvider){

            this.accessToken            = accessToken;
            this.accessTokenSignature   = accessSignature;
            this.dataProvider           = dataProvider;
            this.urlEndpoint            = FlybitsAPIConstants.API_LOGIN_SIGNED;
            type                        = FilterBy.LOGIN_WITH_OTHER;

            return new Options(this);
        }

        /**
         * Login mechanism for logging the user into the Flybits Core using an email and password.
         *
         * @param username The Flybits username.
         * @param password The Flybits password.
         * @return A {@link LoginOptions.Builder.Options} object
         * where additional login settings can be set.
         */
        public Options loginViaEmail(String username, String password){
            this.username       = username;
            this.password       = password;
            this.urlEndpoint    = FlybitsAPIConstants.API_LOGIN;
            type                = FilterBy.LOGIN_WITH_USERNAME;

            return new Options(this);
        }

        /**
         * Login mechanism for logging the user into the Flybits Core anonymously.
         *
         * @return A {@link LoginOptions.Builder.Options} object
         * where additional login settings can be set.
         */
        public Options loginAnonymously(){
            type                = FilterBy.LOGIN_ANONYMOUSLY;
            return new Options(this);
        }

        /**
         * The {@code Builder} used class which defines all optional settings that can be added to
         * the Login FilterOptions including:
         * <ul>
         *     <li>Setting the GCM Sender ID</li>
         *     <li>Adding the Device token</li>
         *     <li>Saving the rememeber me token upon successful login</li>
         * </ul>
         */
        public static class Options{

            private FilterBy type;
            private String username;
            private String password;
            private String accessToken;
            private SocialMediaType socialMediaType;
            private String accessTokenSignature;
            private String dataProvider;
            private boolean retrieveRememberMeToken;
            private boolean sdkInt;
            private boolean isJWTSet;
            private String urlEndpoint;

            private Options(Builder options){
                isJWTSet                = true;
                retrieveRememberMeToken = true;
                this.urlEndpoint        = options.urlEndpoint;
                type                    = options.type;

                switch (type){
                    case LOGIN_WITH_OAUTH:
                        this.socialMediaType    = options.socialMediaType;
                        this.accessToken        = options.accessToken;
                        break;
                    case LOGIN_WITH_REMEMBER_ME_TOKEN:
                        this.accessToken        = options.accessToken;
                        this.dataProvider       = options.dataProvider;
                        break;
                    case LOGIN_WITH_USERNAME:
                        this.username   = options.username;
                        this.password   = options.password;
                        break;
                    case LOGIN_WITH_OTHER:
                        this.accessTokenSignature   = options.accessTokenSignature;
                        this.accessToken            = options.accessToken;
                        this.dataProvider           = options.dataProvider;
                        break;
                }
            }

            /**
             * Build the LoginOptions object to be used for constructing the POST request for
             * {@code login(LoginOptions, IRequestCallback)}.
             *
             * @return {@link LoginOptions} object which can be referenced by the SDK to get specific
             * information about the defined Login mechanism.
             */
            public LoginOptions build(){
                return new LoginOptions(this);
            }

            /**
             * Indicates that the JWT Token should not be obtained when the user logs in.
             *
             * @return A {@link LoginOptions.Builder.Options} object
             * where additional login settings can be set.
             */
            public Options disableJWTReturn(){
                isJWTSet = false;
                return this;
            }

            /**
             * Indicates that the rememberMe Token should not be obtained when the user logs in.
             *
             * @return A {@link LoginOptions.Builder.Options} object
             * where additional login settings can be set.
             */
            public Options disableRememberMeReturn(){
                retrieveRememberMeToken = false;
                return this;
            }

            /**
             * Indicates that the Device OS Version should be set in the Login.
             *
             * @return A {@link LoginOptions.Builder.Options} object
             * where additional login settings can be set.
             */
            public Options setDeviceOSVersion(){
                this.sdkInt = true;
                return this;
            }

            /**
             * Indicates that the JWT Token should be obtained when the user logs in.
             *
             * @return A {@link LoginOptions.Builder.Options} object
             * where additional login settings can be set.
             *
             * @deprecated This function has been deprecated as the JWT token is now automatically
             * available once the user logs in. If the application does not want to retrieve the
             * JWT token it can call {@link Options#disableJWTReturn()}.
             */
            @Deprecated
            public Options setJWTToken(){
                isJWTSet = true;
                return this;
            }

            /**
             * Indicates that the remember me token should be saved internally in the event that the
             * login was successful.
             * @return A {@link LoginOptions.Builder.Options} object
             * where additional login settings can be set.

             * @deprecated This function has been deprecated as the rememberMe token is now
             * automatically available once the user logs in. If the application does not want to
             * retrieve the rememberMe token it can call {@link Options#setRememberMeToken()}.
             */
            @Deprecated
            public Options setRememberMeToken(){
                this.retrieveRememberMeToken = true;
                return this;
            }
        }
    }
}
