package com.flybits.commons.library.http;

import android.content.Context;
import android.support.annotation.NonNull;

import com.flybits.commons.library.FlyingConstants;
import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.exceptions.FlybitsDisabledException;
import com.flybits.commons.library.exceptions.InvalidFlybitsRequestException;
import com.flybits.commons.library.logging.LogType;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.internal.ServerResult;
import com.squareup.okhttp.FormEncodingBuilder;
import com.squareup.okhttp.MediaType;
import com.squareup.okhttp.MultipartBuilder;
import com.squareup.okhttp.Request;
import com.squareup.okhttp.RequestBody;

import java.io.IOException;
import java.util.Map;

/**
 *  <p>The {@code PostRequest} class is responsible for the creation of all POST requests that
 *  should be made between the Flybits SDK and the Flybits Core Server.</p>
 *
 *  @author Petar Kramaric
 *  @version 2.0.0
 *  @since 2.0.0
 */
public class PostRequest extends FlybitsHttpRequest{

    private String url;
    private RequestBody body;
    private boolean isGzip;
    private Map<String, String> headers;

    /**
     * Constructor used to define a POST request where the parameters are defined inside the body of
     * the request as a JSON object.
     *
     * @param context The current context of the application.
     * @param url The url pointing to where the POST request should be made.
     * @param bodyParams The JSON raw body with information that should be attached to the POST
     *                   request.
     */
    public PostRequest(@NonNull Context context, boolean gzipRequest, @NonNull final String url, @NonNull String bodyParams, Map<String, String> headers) throws InvalidFlybitsRequestException{
        super(context, gzipRequest);

        if (context == null || url == null || bodyParams == null){
            throw new InvalidFlybitsRequestException("This request is not valid as a non-null variable was entered as null");
        }

        Logger.instance(FlyingConstants.IS_DEBUG)._LOG(TAG_NETWORK, "Post - Url: " + url, LogType.DEBUG);
        Logger.instance(FlyingConstants.IS_DEBUG)._LOG(TAG_NETWORK, "Post - ( " +bodyParams+ ")", LogType.DEBUG);

        constructRequest(context, url, RequestBody.create(MediaType.parse("application/json; charset=utf-8"), bodyParams), gzipRequest, headers);
    }

    /**
     * Constructor used to define a POST request where the parameters are defined through form
     * parameters.
     *
     * @param context The current context of the application.
     * @param url The url pointing to where the POST request should be made.
     * @param formParams The map of form parameters represented as key-value pairs.
     */
    public PostRequest(@NonNull Context context, boolean gzipRequest, @NonNull final String url, Map<String, String> formParams, Map<String, String> headers)  throws InvalidFlybitsRequestException{
        super(context, gzipRequest);

        if (context == null || url == null){
            throw new InvalidFlybitsRequestException("This request is not valid as a non-null variable was entered as null");
        }

        Logger.instance(FlyingConstants.IS_DEBUG)._LOG(TAG_NETWORK, "Post - Url: " + url, LogType.DEBUG);

        FormEncodingBuilder formBody = new FormEncodingBuilder();
        if (formParams != null && formParams.size() > 0) {
            for (Map.Entry<String, String> entry : formParams.entrySet()) {
                Logger.instance(FlyingConstants.IS_DEBUG)._LOG(TAG_NETWORK, "Post - ( "+entry.getKey()+" , "+ entry.getValue()+" )" + url, LogType.DEBUG);
                formBody.add(entry.getKey(), entry.getValue());
            }
        }else{
            formBody.add("", "");
        }
        constructRequest(context, url, formBody.build(),gzipRequest, headers);
    }

    /**
     * Constructor used to define a POST request where the parameters are defined though a multi-part
     * form builder. In most cases, this method should be used when sending files, such as images,
     * from the SDK/application to the Flybits Core Server.
     *
     * @param context The current context of the application.
     * @param url The url pointing to where the POST request should be made.
     * @param formParams The map of form parameters represented as key-value pairs.
     * @param imageKey The key of the multi-part form expecting a file.
     * @param imageFileName The value of the multi-part form expecting a file. Therefore this would
     *                      be the name of the file.
     * @param bitmapData The bitmap byte array representing the image being transferred from the
     *                   device to the Flybits Core.
     */
    public PostRequest(@NonNull Context context, boolean gzipRequest, @NonNull final String url, Map<String, String> formParams, Map<String, String> headers, @NonNull String imageKey, @NonNull String imageFileName, @NonNull byte[] bitmapData) throws InvalidFlybitsRequestException{
        super(context, gzipRequest);

        if (context == null || url == null || imageKey == null || imageFileName == null || bitmapData == null){
            throw new InvalidFlybitsRequestException("This request is not valid as a non-null variable was entered as null");
        }


        Logger.instance(FlyingConstants.IS_DEBUG)._LOG(TAG_NETWORK, "Post - Url: " + url, LogType.DEBUG);

        MultipartBuilder formBody = new MultipartBuilder();
        if (formParams != null && formParams.size() > 0){
            for (Map.Entry<String, String> entry : formParams.entrySet()) {
                Logger.instance(FlyingConstants.IS_DEBUG)._LOG(TAG_NETWORK, "Post - ( "+entry.getKey()+" , "+ entry.getValue()+" )" + url, LogType.DEBUG);
                formBody.addFormDataPart(entry.getKey(), entry.getValue());
            }
        }

        Logger.instance(FlyingConstants.IS_DEBUG)._LOG(TAG_NETWORK, "Post - ( "+imageKey+" , "+ imageFileName+" )" + url, LogType.DEBUG);
        formBody.addFormDataPart(imageKey, imageFileName, RequestBody.create(MediaType.parse("image/png"), bitmapData));
        constructRequest(context, url, formBody.type(MultipartBuilder.FORM).build(), gzipRequest, headers);
    }

    private void constructRequest(Context context, String url, RequestBody body, boolean isGzip, Map<String, String> headers){

        this.url            = url;
        this.body           = body;
        this.isGzip         = isGzip;
        this.headers        = headers;

        Request.Builder request = setupRequest(context, url);
        request.post(body);

        if (headers != null){
            for (Map.Entry<String, String> entry : headers.entrySet()){
                request.addHeader(entry.getKey(), entry.getValue());
                Logger.instance(FlyingConstants.IS_DEBUG)._LOG(TAG_NETWORK, "Post - ( "+entry.getKey()+" , "+entry.getValue()+" )" + url, LogType.DEBUG);
            }
        }
        setRequest(request.build());
    }

    @Override
    ServerResult refreshJWT(Context context) throws IOException, FlybitsDisabledException {

        if (headers != null){
            headers.put("X-Authorization", SharedElements.getSavedJWTToken(context));
        }

        constructRequest(context, url, body, isGzip, headers);
        return getResponse();
    }
}
