package com.flybits.commons.library.http;

import android.content.Context;

import com.flybits.commons.library.FlyingConstants;
import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.exceptions.FlybitsDisabledException;
import com.flybits.commons.library.logging.LogType;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.internal.ServerResult;
import com.squareup.okhttp.Request;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Map;

/**
 *  <p>The {@code GetRequest} class is responsible for the creation of all GET requests that should
 *  be made between the Flybits SDK and the Flybits Core Server.</p>
 *
 *  @author Petar Kramaric
 *  @version 1.0.0
 *  @since 1.0.0
 */
public class GetRequest extends FlybitsHttpRequest {

    private String url;
    private Map<String, String> headers;

    /**
     * Constructor used to initialize a GET request based on a url.
     *
     * @param context The current state of the application.
     * @param url The url for the GET request.
     * @throws UnsupportedEncodingException If the parameters cannot be encoded because the device
     * does not support UTF-8 encoding.
     */
    public GetRequest(Context context, String url) throws UnsupportedEncodingException {
        this(context, url, null, null);
    }

    /**
     * Constructor used to initialize a GET request based on a url.
     *
     * @param context The current state of the application.
     * @param url The url for the GET request.
     * @throws UnsupportedEncodingException If the parameters cannot be encoded because the device
     * does not support UTF-8 encoding.
     */
    public GetRequest(Context context, String url, Map<String, String> headers) throws UnsupportedEncodingException {
        this(context, url, null, headers);
    }

    /**
     * Constructor used to initialize a GET request with a url and GET url parameters.
     *
     * @param context The current state of the application.
     * @param url The url for the GET request.
     * @param formParams The parameters that should be added to the GET url in the event that the
     *                   url accepts different options.
     * @throws UnsupportedEncodingException If the parameters cannot be encoded because the device
     * does not support UTF-8 encoding.
     */
    public GetRequest(Context context, String url, Map<String, ArrayList<String>> formParams, Map<String, String> headers) throws UnsupportedEncodingException {

        super(context, false);
        this.headers    = headers;

        Logger.instance(FlyingConstants.IS_DEBUG)._LOG(TAG_NETWORK, "Get - Url: " + url, LogType.DEBUG);
        if (formParams != null && formParams.size() > 0) {
            url += "?";

            for (Map.Entry<String, ArrayList<String>> entries : formParams.entrySet()) {
                {
                    for (String entry : entries.getValue()) {
                        if (entries.getKey() == null || entries.getKey().length() == 0){
                            url += entry;
                        }else {
                            url += entries.getKey() + "=" + URLEncoder.encode(entry, "UTF-8");
                        }
                        url += "&";
                    }
                }
            }
            url = url.substring(0, url.length()-1);
        }

        constructRequest(context, url, headers);
    }

    private void constructRequest(Context context, String url, Map<String, String> headers){
        this.url        = url;
        this.headers    = headers;
        Request.Builder request = setupRequest(context, url);

        if (headers != null){
            for (Map.Entry<String, String> entry : headers.entrySet()){
                request.addHeader(entry.getKey(), entry.getValue());
                Logger.instance(FlyingConstants.IS_DEBUG)._LOG(TAG_NETWORK, "Get - ( "+entry.getKey()+" , "+entry.getValue()+" )", LogType.DEBUG);
            }
        }

        setRequest(request.build());
    }

    @Override
    ServerResult refreshJWT(Context context) throws IOException, FlybitsDisabledException {

        if (headers != null){
            headers.put("X-Authorization", SharedElements.getSavedJWTToken(context));
        }

        constructRequest(context, url, headers);
        return getResponse();
    }
}
