package com.flybits.commons.library.http;

import android.content.Context;

import com.flybits.commons.library.api.FlyJWT;
import com.flybits.commons.library.api.FlybitsAPIConstants;
import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.deserializations.DeserializeServerResult;
import com.flybits.commons.library.exceptions.FlybitsDisabledException;
import com.flybits.commons.library.exceptions.InvalidFlybitsResponseException;
import com.flybits.commons.library.logging.LogType;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.JWTToken;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.commons.library.models.internal.ServerResult;
import com.flybits.commons.library.utils.Utilities;
import com.squareup.okhttp.Headers;
import com.squareup.okhttp.OkHttpClient;
import com.squareup.okhttp.Request;
import com.squareup.okhttp.Response;

import java.io.IOException;

import static com.flybits.commons.library.FlyingConstants.IS_DEBUG;

/**
 *  <p>The {@code FlybitsHttpRequest} class is the root class which handles all HTTP(S)
 *  communication between the Flybits SDK and the Flybits Core Server. The main function of this
 *  class is to retrieve the HTTP response from the Flybits Core.</p>
 *
 *  @author Petar Kramaric
 *  @version 2.0.0
 *  @since 2.0.0
 */
abstract class FlybitsHttpRequest {

    private final int MAXTIMEOUT                    = 1;
    private int timeout;

    /**
     * Log tag for all Network traffic to be displayed in the logcat.
     */
    public static final String TAG_NETWORK = "Flybits_Network";

    private final int FLYBITS_DISABLE_CODE          = 503;
    private final int UNAUTHORIZED                  = 401;
    private final String HEADER_X_AUTHENTICATION    = "X-Authorization";
    private final String FLYBITS_X_AUTHENTICATION   = "X-Flybits-Authenticated";

    private OkHttpClient client;
    private Request request;
    private Context mContext;

    private GzipRequestInterceptor gzipInterceptor;

    /**
     * Constructor that accepts the context of the application. This context is used to initialize
     * the Flybits singleton the event that it has not been initiated.
     *
     * @param context The current state of the application.
     */
    @Deprecated
    public FlybitsHttpRequest(Context context){
       this(context, false);
    }

    /**
     * Constructor that accepts the context of the application. This context is used to initialize
     * the Flybits singleton the event that it has not been initiated.
     *
     * @param context The current state of the application.
     */
    public FlybitsHttpRequest(Context context, boolean enableGzip){
        this.mContext           = context;
        this.timeout            = 0;
        this.client             = HttpHelper.getClient(context);

        if (enableGzip) {
            gzipInterceptor = new GzipRequestInterceptor();
            enableGzip();
        }
    }

    protected void setRequest(Request request){
        this.request = request;
    }

    protected  Request.Builder setupRequest(Context context, String url){

        Request.Builder request = new Request.Builder()
                .url(url)
                .header("User-Agent", Utilities.getUserAgentAsJSON())
                .header("Accept", "application/json")
                .header("Content-Type", "application/json")
                .header("flybits-sdk-version", Utilities.getSDKVersion())
                .header("physicalDeviceId", Utilities.getDeviceID(context));


        String apiKey           = FlybitsAPIConstants.getAPIKey(context);
        if (apiKey != null){
            request.header("ApiKey", apiKey);
        }

        if (SharedElements.getEnabledLanguagesAsString(context).length() > 0){
            request.addHeader("Accept-Language", SharedElements.getEnabledLanguagesAsString(context));
        }

        String jwtToken = SharedElements.getSavedJWTToken(context);
        if (jwtToken != null && jwtToken.length() > 0){
            request.addHeader(HEADER_X_AUTHENTICATION, SharedElements.getSavedJWTToken(context) );
        }

        return request;
    }

    protected void enableGzip(){
        Logger.instance(IS_DEBUG)._LOG(TAG_NETWORK, "Interceptor Size: " + client.interceptors().size(), LogType.INFO);
        client.interceptors().add(gzipInterceptor);
    }

    protected void disableGzip(){
        if (client.interceptors().size() > 0) {
            client.interceptors().clear();
        }
    }

    /**
     * Get the response for the GET/POST Request defined in either the {@link GetRequest} or
     * {@link PostRequest} class.
     *
     * @return The result of the GET/POST Request.
     * @throws IOException If OkHttpClient client cannot process the GET/POST request.
     * @throws FlybitsDisabledException If the Flybits Core is down for maintenance or is not
     * currently online.
     */
    public ServerResult getResponse() throws IOException, FlybitsDisabledException {

        final ServerResult result = new ServerResult();
        long timeStart = System.currentTimeMillis();

        Response response = client.newCall(request).execute();

        result.status = response.code();
        if (response.message() != null) {
            result.status_message = response.message();
        }

        Logger.instance(IS_DEBUG)._LOG(TAG_NETWORK, "Response - Time to Process Request: " + (System.currentTimeMillis() - timeStart), LogType.DEBUG);
        Logger.instance(IS_DEBUG)._LOG(TAG_NETWORK, "Response Code: " + result.status, LogType.DEBUG);

        result.response = response.body().string();

        if (result.status == FLYBITS_DISABLE_CODE){

            Result info = new DeserializeServerResult().fromJson(result.response);
            if (info.exception != null && info.exception.getMessage() != null){
                throw new FlybitsDisabledException(info.exception.getMessage());
            }
            throw new FlybitsDisabledException("Flybits is currently not available");
        }

        try {
            Logger.instance(IS_DEBUG)._LOG(TAG_NETWORK, "Response - Result: " + result.response.replace("\r\n", ""), LogType.DEBUG);
        }catch (Exception e){
            Logger.instance(IS_DEBUG)._LOG_EXCEPTION(e);
        }

        if (result.status == UNAUTHORIZED){

            if (timeout < MAXTIMEOUT){
                timeout ++;
                try {
                    Result<JWTToken> refreshResult = FlyJWT.refreshJWT(mContext);
                    if (refreshResult.responseStatus == RequestStatus.SUCCESS){
                        return refreshJWT(mContext);
                    }
                }catch (InvalidFlybitsResponseException e){
                    Logger.instance(IS_DEBUG)._LOG_EXCEPTION(e);
                }
            }
        }

        Headers responseHeaders = response.headers();
        for (int i = 0; i < responseHeaders.size(); i++) {

            if (responseHeaders.name(i).equalsIgnoreCase(HEADER_X_AUTHENTICATION)){
                SharedElements.setJWTToken(mContext, responseHeaders.value(i));
            }
            if (responseHeaders.name(i).equalsIgnoreCase(FLYBITS_X_AUTHENTICATION)){
                SharedElements.setFlybitsXAuthentication(mContext, Boolean.valueOf(responseHeaders.value(i)));
            }
        }

        return result;
    }

    abstract ServerResult refreshJWT(Context context) throws IOException, FlybitsDisabledException;
}
