package com.flybits.commons.library.http;

import android.content.Context;

import com.flybits.commons.library.FlyingConstants;
import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.exceptions.FlybitsDisabledException;
import com.flybits.commons.library.logging.LogType;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.internal.ServerResult;
import com.squareup.okhttp.MediaType;
import com.squareup.okhttp.Request;
import com.squareup.okhttp.RequestBody;

import java.io.IOException;
import java.util.Map;

/**
 *  <p>The {@code DeleteRequest} class is responsible for the creation of all DELETE requests that
 *  should be made between the Flybits SDK and the Flybits Core Server.</p>
 *
 *  @author Petar Kramaric
 *  @version 1.0.0
 *  @since 1.0.0
 */
public class DeleteRequest extends FlybitsHttpRequest{

    /**
     * Defines the MediaType as JSON for the raw body contents.
     */
    public static final MediaType CONTENT_TYPE_JSON = MediaType.parse("application/json; charset=utf-8");

    /**
     * Constructor used to define a DELETE request where the parameters are defined inside the body of
     * the request as a JSON object.
     *
     * @param context The current context of the application.
     * @param url The url pointing to where the DELETE request should be made.
     */
    public DeleteRequest(Context context, final String url)  {
        this(context, false, url, null);
    }

    /**
     * Constructor used to define a DELETE request where the parameters are defined inside the body of
     * the request as a JSON object.
     *
     * @param context The current context of the application.
     * @param url The url pointing to where the DELETE request should be made.
     */
    public DeleteRequest(Context context, final String url, Map<String, String> headers)  {
        this(context, false, url, headers);
    }

    /**
     * Constructor used to define a DELETE request where the parameters are defined inside the body of
     * the request as a JSON object.
     *
     * @param context The current context of the application.
     *
     * @param url The url pointing to where the DELETE request should be made.
     */
    public DeleteRequest(Context context, boolean gzipRequest, final String url, Map<String, String> headers)  {
        super(context, gzipRequest);

        Logger.instance(FlyingConstants.IS_DEBUG)._LOG(TAG_NETWORK, "Delete - Url: " + url, LogType.DEBUG);
        constructRequest(context, url, null, gzipRequest, headers);
    }

    /**
     * Constructor used to define a DELETE request where the parameters are defined inside the body of
     * the request as a JSON object.
     *
     * @param context The current context of the application.
     * @param url The url pointing to where the DELETE request should be made.
     * @param bodyParams The JSON raw body with information that should be attached to the DELETE
     *                   request.
     * @param type Indicates the type of body information that can be found in the DELETE request. In
     *             most cases this will be a JSON MediaType.
     */
    public DeleteRequest(Context context, boolean gzipRequest, final String url, String bodyParams, MediaType type, Map<String, String> headers)  {
        super(context,gzipRequest);

        Logger.instance(FlyingConstants.IS_DEBUG)._LOG(TAG_NETWORK, "Delete - Url: " + url, LogType.DEBUG);
        Logger.instance(FlyingConstants.IS_DEBUG)._LOG(TAG_NETWORK, "Delete - ( " +bodyParams+ ")", LogType.DEBUG);

        if (bodyParams == null || type == null){
            constructRequest(context, url, null, gzipRequest, headers);
        }else {
            constructRequest(context, url, RequestBody.create(type, bodyParams), gzipRequest, headers);
        }
    }


    private String url;
    private RequestBody body;
    private boolean isGzip;
    private Map<String, String> headers;

    private void constructRequest(Context context, String url, RequestBody body, boolean isGzip, Map<String, String> headers){

        this.url            = url;
        this.body           = body;
        this.isGzip         = isGzip;
        this.headers        = headers;

        Request.Builder request = setupRequest(context, url);

        if (body != null) {
            request.delete(body);
        }else{
            request.delete();
        }

        if (headers != null){
            for (Map.Entry<String, String> entry : headers.entrySet()){
                request.addHeader(entry.getKey(), entry.getValue());
                Logger.instance(FlyingConstants.IS_DEBUG)._LOG(TAG_NETWORK, "Delete - ( "+entry.getKey()+" , "+entry.getValue()+" )", LogType.DEBUG);
            }
        }
        setRequest(request.build());
    }

    @Override
    ServerResult refreshJWT(Context context) throws IOException, FlybitsDisabledException {

        if (headers != null){
            headers.put("X-Authorization", SharedElements.getSavedJWTToken(context));
        }

        constructRequest(context, url, body, isGzip, headers);
        return getResponse();
    }
}
